<?php
// vim: set ts=4 sw=4 sts=4 et:

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Module\XC\CustomProductTabs\View\ItemsList\Model;

use XLite\Core\Database;
use XLite\Model\Product\GlobalTab;
use XLite\Model\Product\GlobalTabProvider;

/**
 * GlobalTabs items list
 */
class GlobalTabs extends \XLite\View\ItemsList\Model\Table
{
    /**
     * Get a list of CSS files
     *
     * @return array
     */
    public function getCSSFiles()
    {
        $list = parent::getCSSFiles();
        $list[] = 'modules/XC/CustomProductTabs/global_tabs/style.css';

        return $list;
    }

    /**
     * Define columns structure
     *
     * @return array
     */
    protected function defineColumns()
    {
        return [
            'enabled' => [
                static::COLUMN_CLASS => 'XLite\View\FormField\Inline\Input\Checkbox\Switcher\ShowHide',
            ],
            'name'    => [
                static::COLUMN_NAME => \XLite\Core\Translation::lbl('Name'),
                static::COLUMN_MAIN => true,
                static::COLUMN_LINK => true,
            ],
        ];
    }

    protected function getRightActions()
    {
        return array_merge(parent::getRightActions(), [
            'modules/XC/CustomProductTabs/global_tabs/help.twig'
        ]);
    }

    /**
     * Return true if 'Edit' link should be displayed in column line
     *
     * @param array                $column
     * @param \XLite\Model\AEntity $entity
     *
     * @return boolean
     */
    protected function isLink(array $column, \XLite\Model\AEntity $entity)
    {
        return parent::isLink($column, $entity) && $entity->getCustomTab();
    }

    /**
     * Build entity page URL
     *
     * @param \XLite\Model\AEntity $entity Entity
     * @param array                $column Column data
     *
     * @return string
     */
    protected function buildEntityURL(\XLite\Model\AEntity $entity, array $column)
    {
        return \XLite\Core\Converter::buildURL(
            'global_tab',
            '',
            ['tab_id' => $entity->getId()]
        );
    }

    /**
     * Define repository name
     *
     * @return string
     */
    protected function defineRepositoryName()
    {
        return 'XLite\Model\Product\GlobalTab';
    }

    /**
     * Creation button position
     *
     * @return integer
     */
    protected function isCreation()
    {
        return static::CREATE_INLINE_TOP;
    }

    /**
     * Get create entity URL
     *
     * @return string
     */
    protected function getCreateURL()
    {
        return \XLite\Core\Converter::buildURL(
            'global_tab',
            '',
            ['tab_id' => 0]
        );
    }

    /**
     * Get create button label
     *
     * @return string
     */
    protected function getCreateButtonLabel()
    {
        return 'New tab';
    }


    // {{{ Behaviors

    /**
     * Mark list as removable
     *
     * @return boolean
     */
    protected function isRemoved()
    {
        return true;
    }

    /**
     * Check - remove entity or not
     *
     * @param \XLite\Model\AEntity $entity Entity
     *
     * @return boolean
     */
    protected function isAllowEntityRemove(\XLite\Model\AEntity $entity)
    {
        return parent::isAllowEntityRemove($entity) && $entity->getCustomTab();
    }

    /**
     * @param array                $column
     * @param \XLite\Model\AEntity $model
     *
     * @return null|string
     */
    public function getHelpText(array $column, \XLite\Model\AEntity $model)
    {
        if (
            $column['code'] === 'actions right'
            && !$model->getCustomTab()
        ) {
            return $this->getTabHelpText($model);
        }

        return null;
    }

    /**
     * @param \XLite\Module\XC\CustomProductTabs\Model\Product\GlobalTab $model
     *
     * @return string
     */
    protected function getTabHelpText(GlobalTab $model)
    {
        if ($model->getServiceName() === 'Description') {
            return static::t('Tab displaying the product\'s detailed description. Added by the X-Cart core');
        }

        if ($model->getServiceName() === 'Specification') {
            return static::t('Tab displaying the product\'s attributes and other details. Added by the X-Cart core');
        }

        if ($model->getServiceName() === 'Comments') {
            return static::t('Tab displaying comments about the product. Added by the addons VK/GoSocial/Disqus', [
                'modules' => implode(', ', array_filter(array_map(function(GlobalTabProvider $provider) {
                    $code = $provider->getCode();
                    $repo = Database::getRepo('XLite\Model\Module');

                    if ($module = $repo->findOneByModuleName($code)) {
                        return sprintf(
                            '<a href="%s">%s</a>',
                            $module->isInstalled() ? $module->getInstalledURL(): $module->getMarketplaceURL(),
                            $module->getName()
                        );
                    }

                    return null;
                }, $model->getProviders()->toArray())))
            ]);
        }

        return '';
    }

    /**
     * Mark list as sortable
     *
     * @return integer
     */
    protected function getSortableType()
    {
        return static::SORT_TYPE_MOVE;
    }

    // }}}

    /**
     * Get container class
     *
     * @return string
     */
    protected function getContainerClass()
    {
        return parent::getContainerClass() . ' global-tabs';
    }

    /**
     * Get panel class
     *
     * @return string|\XLite\View\Base\FormStickyPanel
     */
    protected function getPanelClass()
    {
        return 'XLite\Module\XC\CustomProductTabs\View\StickyPanel\ItemsList\GlobalTab';
    }
}