<?php
// vim: set ts=4 sw=4 sts=4 et:

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XLite\Console\Command\Scaffolding;

use Includes\Utils\FileManager;
use Includes\Utils\ModulesManager;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\ArrayInput;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use XLite\Console\Command\Exception\ConfigurationError;
use XLite\Console\Command\Helpers;
use XLite\Console\Command\SourceCodeGenerators;
use XLite\Console\Command\SourceCodeGenerators\PhpClass;
use XLite\Console\Command\SourceCodeGenerators\Utils;
use XLite\Core\Converter;

class ItemsList extends Command
{
    use Helpers\ModuleTrait;
    use Helpers\ControllerTrait;

    protected function configure()
    {
        $this
            ->setName('scaffolding:itemsList')
            ->setDescription('Generate an ItemsList and related files for the given entity.')
            ->setHelp('This command generates the items list page for the common CRUD operations on the given entity')

            ->addArgument('entity', InputArgument::REQUIRED, 'Entity class name. Example - XLite\\\\Model\\\\Product')
            ->addOption('module', 'm', InputOption::VALUE_REQUIRED, 'Put generated files in the module dir. Example - authorName\\\\moduleName')
            ->addOption('fields', 'f', InputOption::VALUE_REQUIRED, 'Comma-separated fields / columns list. Example - value1,value2,value3')
            ->addOption('searchFields', 'F', InputOption::VALUE_REQUIRED, 'Comma-separated search fields list. Example - value1,value2,value3. Default - without any search.')
            ->addOption('target', 't', InputOption::VALUE_REQUIRED, 'List controller target. Default is the pluralized entity name (\XLite\Model\Product -> products)')
            ->addOption('itemsListStyle', 'T', InputOption::VALUE_NONE, 'Create a default style file template')

            ->addOption('removable', 'r', InputOption::VALUE_NONE, 'Allow to remove items')
            ->addOption('switchable', 's', InputOption::VALUE_NONE, 'Allow to switch items state (enable\disable)')
            ->addOption('sortable', 'S', InputOption::VALUE_NONE, 'Allow to sort items within the list')
            ->addOption('create', 'C', InputOption::VALUE_NONE, 'Allow to create new items')
            ->addOption('edit', 'e', InputOption::VALUE_NONE, 'Allow to edit the items')

            ->addOption('editLinkColumn', 'E', InputOption::VALUE_REQUIRED, 'Edit link column name', false)
            ->addOption('inlineCreate', 'I', InputOption::VALUE_NONE, 'Allow to inline create the items')
            ->addOption('editFields', 'D', InputOption::VALUE_REQUIRED, 'Inline editable fields list. Example - value1,value2,value3. By default all columns are read-only.')

            ->addOption('menu', 'M', InputOption::VALUE_REQUIRED, 'Create link in the left menu. Works only if the "module" option is provided. Examples: admin area - "main.users", "bottom.store_setup", customer area - "{my account}", "my_awesome_page')

            ->addOption('rebuildAfterwards', 'R', InputOption::VALUE_NONE, 'Recalculate view lists afterwards')
        ;
    }

    protected function prepareOptions(InputInterface $input, SymfonyStyle $io)
    {
        $entityClass = $input->getArgument('entity');
        if (!\XLite\Core\Database::getRepo($entityClass)) {
            throw new ConfigurationError('We couldn\'t find model ' . $entityClass);
        }

        $shortModelName = Utils::getClassShortName($entityClass);

        $target = $input->getOption('target')
            ?: str_replace('\\', '_', $shortModelName) . 's';
        $target = strtolower($target);

        $zones = $this->isControllerExistsInZones($target);

        if ($zones['Admin'] === true
            && !$io->confirm("Target '$target' is already in use, found in Admin zone. Overwrite?", false)
        ) {
            if ($input->getOption('target')) {
                throw new ConfigurationError(
                    "Target '$target' is already in use, found in Admin zone. Try another target"
                );
            }

            throw new ConfigurationError(
                "Autogenerated target '$target' is already in use in Admin zone. Try to provide a target via --target option"
            );
        }

        $module = $input->getOption('module')
            ?: null;

        if ($module) {
            try {
                $state = $this->getModuleStateByName($module);

                if ($state === Helpers\Module::NOT_FOUND) {
                    $io->warning("Module $module does not exists.");
                } elseif ($state === Helpers\Module::NOT_INSTALLED) {
                    $io->warning("Module $module already exists, but bot installed");
                }

            } catch(\Exception $e) {
                $io->warning($e->getMessage());
            }
        }

        $rootNamespace = $module
            ? 'XLite\Module\\' . $module . '\\'
            : 'XLite\\';

        $rootPath = str_replace('/', LC_DS, $rootNamespace);

        $skinsRootPath = $module
            ? 'admin/' . str_replace('\\', '/', $module) . '/'
            : 'admin/';

        $fieldsRaw = $input->getOption('fields') ?: '';
        $searchFieldsRaw = $input->getOption('searchFields') ?: '';
        $editFieldsRaw = $input->getOption('editFields') ?: '';

        $fields = explode(',', $fieldsRaw);
        $searchFields = explode(',', $searchFieldsRaw);
        $editFields = $editFieldsRaw
            ? explode(',', $editFieldsRaw)
            : $fields;

        $repo = \XLite\Core\Database::getRepo($entityClass);

        $fallbackEditColumn = $repo
            ? $repo->getPrimaryKeyField()
            : '';

        $editLinkColumn = $input->getOption('editLinkColumn') ?: $fallbackEditColumn;

        $resultFields = [];

        foreach ($fields as $fieldName) {
            $fieldProcessed = [
                'name'      => $fieldName,
                'humanName' => Utils::convertCamelToHumanReadable($fieldName),
            ];

            $fieldProcessed['isEdit'] = $editLinkColumn === $fieldName;
            $fieldProcessed['isSearch'] = in_array($fieldName, $searchFields, true);

            $resultFields[] = $fieldProcessed;
        }

        $resultEditFields = [];

        foreach ($editFields as $fieldName) {
            $fieldProcessed = [
                'name'      => $fieldName,
                'humanName' => Utils::convertCamelToHumanReadable($fieldName),
            ];

            $resultEditFields[] = $fieldProcessed;
        }

        if ($input->getOption('sortable')) {
            $io->warning("You've allowed to sort the items within the list. Please note that the column sorting will be disabled.");
        }

        $leftMenuPath = $input->getOption('menu');

        if ($leftMenuPath === 'none') {
            $leftMenuPath = $io->ask('Left menu path ("main.users" or "bottom.look")', null);
        }

        if ($leftMenuPath
            && substr_count($leftMenuPath, '.') + 1 !== 2
        ) {
            throw new ConfigurationError(
                'Left menu path has wrong format: ' . $leftMenuPath
            );
        }

        if ($leftMenuPath && !$module) {
            $io->warning("Can't generate the left menu link because the items list files are generated in the core.");
            $leftMenuPath = null;
        }

        return [
            'entityClass'       => $entityClass,
            'shortModelName'    => $shortModelName,
            'target'            => $target,
            'module'            => $module,
            'rootPath'          => $rootPath,
            'rootNamespace'     => $rootNamespace,
            'skinsRootPath'     => $skinsRootPath,
            'viewModelStyle'    => $input->getOption('itemsListStyle'),
            'fields'            => $resultFields,
            'searchFields'      => $searchFields,
            'rawEditFields'     => $editFields,
            'editFields'        => $resultEditFields,
            'leftMenuPath'      => $leftMenuPath,
            'removable'         => $input->getOption('removable'),
            'switchable'        => $input->getOption('switchable'),
            'sortable'          => $input->getOption('sortable'),
            'canCreate'         => $input->getOption('create'),
            'canEdit'           => $input->getOption('edit'),
            'inlineCreation'    => $input->getOption('inlineCreate'),
            'editColumn'        => $editLinkColumn,
            'createTarget'      => $target . '_edit',
            'editTarget'        => $target . '_edit',
            'rebuildAfterwards' => $input->getOption('rebuildAfterwards'),
            'url'                  => \XLite\Core\URLManager::getShopURL(
                \XLite\Core\Converter::buildURL($target, '', [], \XLite::getAdminScript())
            ),
        ];
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $io = new SymfonyStyle($input, $output);

        $io->title('Scaffolding: ItemsList');

        try {
            $config = $this->prepareOptions($input, $io);
        } catch (ConfigurationError $e) {
            $io->error($e->getMessage());
            return;
        }

        list($path, $controllerClass) = $this->createController($config, $io);

        if ($path) {
            if (LC_DEVELOPER_MODE && class_exists($controllerClass, true)) {
                new $controllerClass();
            }
            $io->note($path .' itemsList controller is generated');
        }

        if ($config['canEdit'] || ($config['canCreate'] && !$config['inlineCreation'])) {
            $this->createFormModelInfrastructure($input, $output, $config);
        }


        list($path, $viewClass) = $this->createViewModel($config, $io);

        if ($path) {
            if (LC_DEVELOPER_MODE && class_exists($viewClass, true)) {
                new $viewClass();
            }
            $io->note($path .' itemsList view is generated');
        }

        if ($config['leftMenuPath']) {
            list($path, $leftMenuClass) = $this->createLeftMenuClass($config, $io);

            if ($path) {
                if (LC_DEVELOPER_MODE && class_exists($leftMenuClass, true)) {
                    new $leftMenuClass();
                }
                $io->note($path .' left menu decorator is generated');
            }
        }

        if ($config['rebuildAfterwards']) {
            $this->rebuildViewLists($output);
        }

        $io->success('ItemsList scaffolding complete');
        $io->text('<info>You can see the result</info>: ' . $config['url']);
    }

    /**
     * @param array        $config
     * @param SymfonyStyle $io
     *
     * @return array
     */
    protected function createController(array $config, SymfonyStyle $io)
    {
        $controllerGenerator = new SourceCodeGenerators\ItemsList\Controller(
            $this->createBaseGenerator()
        );

        $name = Converter::convertToCamelCase($config['target']);
        $namespace =  $config['rootNamespace'] . 'Controller\Admin';
        $fqn = '\\' . $namespace . '\\' . $name;

        if (class_exists($fqn, true) && !$io->confirm("Class $fqn already exists. Overwrite?")) {
            return [ false, $fqn ];
        }

        $content = $controllerGenerator->generate(
            $name,
            $namespace,
            $config['module']
        );

        return [
            Utils::saveClass($fqn, $content),
            $fqn
        ];
    }


    /**
     * @param array        $config
     * @param SymfonyStyle $io
     *
     * @return array
     */
    protected function createLeftMenuClass(array $config, SymfonyStyle $io)
    {
        $generator = new SourceCodeGenerators\Menu\LeftMenuDecorator(
            $this->createBaseGenerator()
        );

        $name = Converter::convertToCamelCase($config['target']) . 'TopMenu';
        $namespace =  $config['rootNamespace'] . 'View\Menu\Admin';
        $fqn = '\\' . $namespace . '\\' . $name;

        if (class_exists($fqn, true) && !$io->confirm("Class $fqn already exists. Overwrite?")) {
            return [ false, $fqn ];
        }

        $content = $generator->generate(
            $name,
            $namespace,
            'Autogenerated ' . $config['shortModelName'] . ' menu item',
            $config['leftMenuPath'],
            $config['target']
        );

        return [
            Utils::saveClass($fqn, $content),
            $fqn
        ];
    }

    protected function createViewModel(array $config, SymfonyStyle $io)
    {
        $viewModelGenerator =  new SourceCodeGenerators\ItemsList\ModelView(
            $this->createBaseGenerator()
        );
        $viewModelGenerator->setIsRemovable($config['removable']);
        $viewModelGenerator->setIsSwitchable($config['switchable']);
        $viewModelGenerator->setIsSortable($config['sortable']);

        $name = $config['shortModelName']. 'ItemsList';
        $namespace =  $config['rootNamespace'] . 'View\ItemsList\Model';
        $fqn = '\\' . $namespace . '\\' . $name;

        if ($config['viewModelStyle']) {
            $templatesPath = LC_DIR_CLASSES . 'XLite/Console/Command/SourceCodeGenerators/templates/';
            $from = $templatesPath . 'base/base-style.less';
            $pathPart = 'items_list/scaffolded_' . strtolower($config['shortModelName']) . '.less';
            $path = LC_DIR_SKINS . $config['skinsRootPath'] . $pathPart;

            FileManager::copy(
                str_replace('/', LC_DS, $from),
                str_replace('/', LC_DS, $path)
            );

            $viewModelGenerator->setStylePath($pathPart);
        }

        $viewModelGenerator->setCanCreate($config['canCreate']);
        $viewModelGenerator->setCanEdit($config['canEdit']);
        $viewModelGenerator->setInlineCreation($config['inlineCreation']);
        $viewModelGenerator->setCreateTarget($config['createTarget']);
        $viewModelGenerator->setEditTarget($config['editTarget']);

        if (class_exists($fqn, true) && !$io->confirm("Class $fqn already exists. Overwrite?")) {
            return [ false, $fqn ];
        }

        $content = $viewModelGenerator->generate(
            $name,
            $namespace,
            $config['entityClass'],
            $config['module'],
            $config['target'],
            $config['fields']
        );

        return [
            Utils::saveClass($fqn, $content),
            $fqn
        ];
    }

    /**
     * @param InputInterface  $parentInput
     * @param OutputInterface $output
     */
    protected function createFormModelInfrastructure(InputInterface $parentInput, OutputInterface $output, $config)
    {
        $command = $this->getApplication()->find('scaffolding:formModel');

        $options = [
            'command'             => $command->getName(),
            'entity'              => $parentInput->getArgument('entity'),
            '--rebuildAfterwards' => false,
            '--fields'            => implode(',', $config['rawEditFields']),
        ];

        if ($parentInput->getOption('module')) {
            $options['--module'] = $parentInput->getOption('module');
        }

        if ($config['editTarget']) {
            $options['--target'] = $config['editTarget'];
        }
        $input = new ArrayInput($options);

        return $command->run($input, $output);
    }

    /**
     * @param OutputInterface $output
     */
    protected function rebuildViewLists(OutputInterface $output)
    {
        $command = $this->getApplication()->find('utils:rebuildViewLists');

        $arguments = [ 'command' => $command->getName() ];

        $input = new ArrayInput($arguments);
        $command->run($input, $output);
    }

    /**
     * @return PhpClass
     */
    protected function createBaseGenerator()
    {
        return new PhpClass(
            new SourceCodeGenerators\Renderer\TwigRenderer()
        );
    }
}
