<?php

/*
 * PEL: PHP Exif Library. A library with support for reading and
 * writing all Exif headers in JPEG and TIFF images using PHP.
 *
 * Copyright (C) 2005, 2006 Martin Geisler.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program in the file COPYING; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA 02110-1301 USA
 */

use lsolesen\pel\Pel;
use lsolesen\pel\PelJpeg;

class PentaxIstDSTest extends \PHPUnit_Framework_TestCase
{
    public function testRead()
    {
        Pel::clearExceptions();
        Pel::setStrictParsing(false);
        $jpeg = new PelJpeg(dirname(__FILE__) . '/pentax-istDS.jpg');

        $exif = $jpeg->getExif();
        $this->assertInstanceOf('lsolesen\pel\PelExif', $exif);

        $tiff = $exif->getTiff();
        $this->assertInstanceOf('lsolesen\pel\PelTiff', $tiff);

        /* The first IFD. */
        $ifd0 = $tiff->getIfd();
        $this->assertInstanceOf('lsolesen\pel\PelIfd', $ifd0);

        /* Start of IDF $ifd0. */
        $this->assertEquals(count($ifd0->getEntries()), 13);

        $entry = $ifd0->getEntry(271); // Make
        $this->assertInstanceOf('lsolesen\pel\PelEntryAscii', $entry);
        $this->assertEquals($entry->getValue(), 'PENTAX Corporation ');
        $this->assertEquals($entry->getText(), 'PENTAX Corporation ');

        $entry = $ifd0->getEntry(272); // Model
        $this->assertInstanceOf('lsolesen\pel\PelEntryAscii', $entry);
        $this->assertEquals($entry->getValue(), 'PENTAX *ist DS     ');
        $this->assertEquals($entry->getText(), 'PENTAX *ist DS     ');

        $entry = $ifd0->getEntry(274); // Orientation
        $this->assertInstanceOf('lsolesen\pel\PelEntryShort', $entry);
        $this->assertEquals($entry->getValue(), 1);
        $this->assertEquals($entry->getText(), 'top - left');

        $entry = $ifd0->getEntry(282); // XResolution
        $this->assertInstanceOf('lsolesen\pel\PelEntryRational', $entry);
        $this->assertEquals($entry->getValue(), array(
            0 => 72,
            1 => 1
        ));
        $this->assertEquals($entry->getText(), '72/1');

        $entry = $ifd0->getEntry(283); // YResolution
        $this->assertInstanceOf('lsolesen\pel\PelEntryRational', $entry);
        $this->assertEquals($entry->getValue(), array(
            0 => 72,
            1 => 1
        ));
        $this->assertEquals($entry->getText(), '72/1');

        $entry = $ifd0->getEntry(296); // ResolutionUnit
        $this->assertInstanceOf('lsolesen\pel\PelEntryShort', $entry);
        $this->assertEquals($entry->getValue(), 2);
        $this->assertEquals($entry->getText(), 'Inch');

        $entry = $ifd0->getEntry(305); // Software
        $this->assertInstanceOf('lsolesen\pel\PelEntryAscii', $entry);
        $this->assertEquals($entry->getValue(), '*ist DS     Ver 1.00   ');
        $this->assertEquals($entry->getText(), '*ist DS     Ver 1.00   ');

        $entry = $ifd0->getEntry(306); // DateTime
        $this->assertInstanceOf('lsolesen\pel\PelEntryTime', $entry);
        $this->assertEquals($entry->getValue(), 1106749970);
        $this->assertEquals($entry->getText(), '2005:01:26 14:32:50');

        $entry = $ifd0->getEntry(318); // WhitePoint
        $this->assertInstanceOf('lsolesen\pel\PelEntryRational', $entry);
        $this->assertEquals($entry->getValue(), array(
            0 => array(
                0 => 313,
                1 => 1000
            ),
            1 => array(
                0 => 329,
                1 => 1000
            )
        ));
        $this->assertEquals($entry->getText(), '313/1000, 329/1000');

        $entry = $ifd0->getEntry(319); // PrimaryChromaticities
        $this->assertInstanceOf('lsolesen\pel\PelEntryRational', $entry);
        $this->assertEquals($entry->getValue(), array(
            0 => array(
                0 => 64,
                1 => 100
            ),
            1 => array(
                0 => 33,
                1 => 100
            ),
            2 => array(
                0 => 21,
                1 => 100
            ),
            3 => array(
                0 => 71,
                1 => 100
            ),
            4 => array(
                0 => 15,
                1 => 100
            ),
            5 => array(
                0 => 6,
                1 => 100
            )
        ));
        $this->assertEquals($entry->getText(), '64/100, 33/100, 21/100, 71/100, 15/100, 6/100');

        $entry = $ifd0->getEntry(529); // YCbCrCoefficients
        $this->assertInstanceOf('lsolesen\pel\PelEntryRational', $entry);
        $this->assertEquals($entry->getValue(), array(
            0 => array(
                0 => 299,
                1 => 1000
            ),
            1 => array(
                0 => 587,
                1 => 1000
            ),
            2 => array(
                0 => 114,
                1 => 1000
            )
        ));
        $this->assertEquals($entry->getText(), '299/1000, 587/1000, 114/1000');

        $entry = $ifd0->getEntry(531); // YCbCrPositioning
        $this->assertInstanceOf('lsolesen\pel\PelEntryShort', $entry);
        $this->assertEquals($entry->getValue(), 2);
        $this->assertEquals($entry->getText(), 'co-sited');

        $entry = $ifd0->getEntry(50341); // PrintIM
        $this->assertInstanceOf('lsolesen\pel\PelEntryUndefined', $entry);
        $expected = "\x50\x72\x69\x6e\x74\x49\x4d\0\x30\x33\x30\x30\0\0\0\x21\0\x01\0\x16\0\x16\0\x02\x01\0\0\0\0\x03\0\0\0\xd6\0\x07\0\0\0\0\0\x08\0\0\0\0\0\x09\0\0\0\0\0\x0a\0\0\0\0\0\x0b\0\0\x01\x1e\0\x0c\0\0\0\0\0\x0d\0\0\0\0\0\x0e\0\0\x01\x34\x01\0\x05\0\0\0\x01\x01\xff\0\0\0\x01\x02\x83\0\0\0\x01\x03\x83\0\0\0\x01\x04\x83\0\0\0\x01\x05\x83\0\0\0\x01\x06\x83\0\0\0\x01\x07\x80\x80\x80\0\x01\x10\x80\0\0\0\x02\0\0\0\0\0\x02\x07\0\0\0\0\x02\x08\0\0\0\0\x02\x09\0\0\0\0\x02\x0a\0\0\0\0\x02\x0b\0\0\x01\x44\x02\x0d\0\0\0\0\x03\0\x05\0\0\0\x03\x01\xff\0\0\0\x03\x02\x83\0\0\0\x03\x03\x83\0\0\0\x03\x06\x83\0\0\0\x03\x10\x80\0\0\0\0\0\x17\xd2\0\0\x27\x10\0\0\x08\x05\0\0\x27\x10\0\0\x05\xd4\0\0\x27\x10\0\0\x0c\x27\0\0\x27\x10\0\0\x18\x71\0\0\x27\x10\0\0\x02\x78\0\0\x27\x10\0\0\0\xc2\0\0\x27\x10\0\0\x02\x61\0\0\x27\x10\0\0\x1d\x18\0\0\x27\x10\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0";
        $this->assertEquals($entry->getValue(), $expected);
        $this->assertEquals($entry->getText(), '(undefined)');

        /* Sub IFDs of $ifd0. */
        $this->assertEquals(count($ifd0->getSubIfds()), 1);
        $ifd0_0 = $ifd0->getSubIfd(2); // IFD Exif
        $this->assertInstanceOf('lsolesen\pel\PelIfd', $ifd0_0);

        /* Start of IDF $ifd0_0. */
        $this->assertEquals(count($ifd0_0->getEntries()), 30);

        $entry = $ifd0_0->getEntry(33434); // ExposureTime
        $this->assertInstanceOf('lsolesen\pel\PelEntryRational', $entry);
        $this->assertEquals($entry->getValue(), array(
            0 => 1,
            1 => 4
        ));
        $this->assertEquals($entry->getText(), '1/4 sec.');

        $entry = $ifd0_0->getEntry(33437); // FNumber
        $this->assertInstanceOf('lsolesen\pel\PelEntryRational', $entry);
        $this->assertEquals($entry->getValue(), array(
            0 => 45,
            1 => 10
        ));
        $this->assertEquals($entry->getText(), 'f/4.5');

        $entry = $ifd0_0->getEntry(34850); // ExposureProgram
        $this->assertInstanceOf('lsolesen\pel\PelEntryShort', $entry);
        $this->assertEquals($entry->getValue(), 2);
        $this->assertEquals($entry->getText(), 'Normal program');

        $entry = $ifd0_0->getEntry(34855); // ISOSpeedRatings
        $this->assertInstanceOf('lsolesen\pel\PelEntryShort', $entry);
        $this->assertEquals($entry->getValue(), 200);
        $this->assertEquals($entry->getText(), '200');

        $entry = $ifd0_0->getEntry(36864); // ExifVersion
        $this->assertInstanceOf('lsolesen\pel\PelEntryVersion', $entry);
        $this->assertEquals($entry->getValue(), 2.21);
        $this->assertEquals($entry->getText(), 'Exif Version 2.21');

        $entry = $ifd0_0->getEntry(36867); // DateTimeOriginal
        $this->assertInstanceOf('lsolesen\pel\PelEntryTime', $entry);
        $this->assertEquals($entry->getValue(), 1106749970);
        $this->assertEquals($entry->getText(), '2005:01:26 14:32:50');

        $entry = $ifd0_0->getEntry(36868); // DateTimeDigitized
        $this->assertInstanceOf('lsolesen\pel\PelEntryTime', $entry);
        $this->assertEquals($entry->getValue(), 1106749970);
        $this->assertEquals($entry->getText(), '2005:01:26 14:32:50');

        $entry = $ifd0_0->getEntry(37121); // ComponentsConfiguration
        $this->assertInstanceOf('lsolesen\pel\PelEntryUndefined', $entry);
        $this->assertEquals($entry->getValue(), "\x01\x02\x03\0");
        $this->assertEquals($entry->getText(), 'Y Cb Cr -');

        $entry = $ifd0_0->getEntry(37380); // ExposureBiasValue
        $this->assertInstanceOf('lsolesen\pel\PelEntrySRational', $entry);
        $this->assertEquals($entry->getValue(), array(
            0 => 0,
            1 => 10
        ));
        $this->assertEquals($entry->getText(), '0.0');

        $entry = $ifd0_0->getEntry(37383); // MeteringMode
        $this->assertInstanceOf('lsolesen\pel\PelEntryShort', $entry);
        $this->assertEquals($entry->getValue(), 5);
        $this->assertEquals($entry->getText(), 'Pattern');

        $entry = $ifd0_0->getEntry(37385); // Flash
        $this->assertInstanceOf('lsolesen\pel\PelEntryShort', $entry);
        $this->assertEquals($entry->getValue(), 16);
        $this->assertEquals($entry->getText(), 'Flash did not fire, compulsory flash mode.');

        $entry = $ifd0_0->getEntry(37386); // FocalLength
        $this->assertInstanceOf('lsolesen\pel\PelEntryRational', $entry);
        $this->assertEquals($entry->getValue(), array(
            0 => 45,
            1 => 1
        ));
        $this->assertEquals($entry->getText(), '45.0 mm');

        $entry = $ifd0_0->getEntry(37500); // MakerNote
        $this->assertInstanceOf('lsolesen\pel\PelEntryUndefined', $entry);
        $expected = "\x41\x4f\x43\0\x4d\x4d\0\x48\0\x01\0\x03\0\0\0\x01\0\0\0\0\0\x02\0\x03\0\0\0\x02\x02\x80\x01\xe0\0\x03\0\x04\0\0\0\x01\0\0\x5c\x04\0\x04\0\x04\0\0\0\x01\0\0\x09\x3c\0\x05\0\x04\0\0\0\x01\0\x01\x2a\xa2\0\x06\0\x07\0\0\0\x04\x07\xd5\x01\x1a\0\x07\0\x07\0\0\0\x03\x0e\x20\x32\0\0\x08\0\x03\0\0\0\x01\0\0\0\0\0\x09\0\x03\0\0\0\x02\0\x22\0\x02\0\x0c\0\x03\0\0\0\x01\0\x01\0\0\0\x0d\0\x03\0\0\0\x01\0\x10\0\0\0\x0e\0\x03\0\0\0\x01\xff\xff\0\0\0\x12\0\x04\0\0\0\x01\0\0\x61\xa8\0\x13\0\x03\0\0\0\x01\0\x2d\0\0\0\x14\0\x03\0\0\0\x01\0\x09\0\0\0\x16\0\x03\0\0\0\x01\0\x32\0\0\0\x17\0\x03\0\0\0\x01\0\0\0\0\0\x18\0\x03\0\0\0\x01\0\0\0\0\0\x19\0\x03\0\0\0\x01\0\x04\0\0\0\x1a\0\x03\0\0\0\x01\xff\xff\0\0\0\x1d\0\x04\0\0\0\x01\0\0\x11\x94\0\x1f\0\x03\0\0\0\x02\0\x01\0\0\0\x20\0\x03\0\0\0\x02\0\x01\0\0\0\x21\0\x03\0\0\0\x02\0\x01\0\0\0\x22\0\x03\0\0\0\x01\0\0\0\0\0\x23\0\x03\0\0\0\x01\0\x16\0\0\0\x24\0\x03\0\0\0\x01\0\x16\0\0\0\x25\0\x03\0\0\0\x01\0\x01\0\0\0\x26\0\x03\0\0\0\x01\0\x01\0\0\0\x27\0\x07\0\0\0\x04\xfe\xff\xff\xff\0\x28\0\x07\0\0\0\x04\xfe\xff\xff\xff\0\x29\0\x04\0\0\0\x01\0\0\x05\x46\0\x2b\0\x04\0\0\0\x01\0\x04\x44\0\0\x2c\0\x04\0\0\0\x01\0\0\0\0\0\x2d\0\x03\0\0\0\x01\x16\x80\0\0\0\x32\0\x07\0\0\0\x04\0\0\0\0\0\x33\0\x01\0\0\0\x03\x02\0\x01\0\0\x34\0\x01\0\0\0\x04\0\0\0\0\0\x35\0\x03\0\0\0\x02\x2e\xec\x1f\x33\0\x36\0\x03\0\0\0\x01\0\x80\0\0\0\x37\0\x03\0\0\0\x01\0\x01\0\0\0\x3a\0\x03\0\0\0\x01\x46\x66\0\0\0\x3d\0\x03\0\0\0\x01\x20\0\0\0\0\x3e\0\x01\0\0\0\x04\x1a\x1a\0\0\0\x3f\0\x01\0\0\0\x02\x04\xfc\0\0\0\x41\0\x03\0\0\0\x01\0\0\0\0\0\x47\0\x06\0\0\0\x01\x11\0\0\0\0\x48\0\x03\0\0\0\x01\0\0\0\0\0\x49\0\x03\0\0\0\x01\0\x01\0\0\0\x4f\0\x03\0\0\0\x01\0\x01\0\0\x02\0\0\x03\0\0\0\x04\0\0\x08\x0c\x02\x01\0\x03\0\0\0\x04\0\0\x08\x14\x02\x02\0\x03\0\0\0\x04\0\0\x08\x1c\x02\x05\0\x07\0\0\0\x10\0\0\x08\x24\x02\x06\0\x07\0\0\0\x0e\0\0\x08\x34\x02\x07\0\x07\0\0\0\x24\0\0\x08\x44\x02\x08\0\x07\0\0\0\x1c\0\0\x08\x68\x02\x09\0\x07\0\0\0\x10\0\0\x08\x84\x02\x0a\0\x07\0\0\0\x10\0\0\x08\x94\x02\x0b\0\x07\0\0\0\x10\0\0\x08\xa4\x02\x0d\0\x03\0\0\0\x04\0\0\x08\xb4\x02\x0e\0\x03\0\0\0\x04\0\0\x08\xbc\x02\x0f\0\x03\0\0\0\x04\0\0\x08\xc4\x02\x10\0\x03\0\0\0\x04\0\0\x08\xcc\x02\x11\0\x03\0\0\0\x04\0\0\x08\xd4\x02\x12\0\x03\0\0\0\x04\0\0\x08\xdc\x02\x13\0\x03\0\0\0\x04\0\0\x08\xe4\x02\x14\0\x03\0\0\0\x04\0\0\x08\xec\x02\x15\0\x04\0\0\0\x05\0\0\x08\xf4\x02\x16\0\x07\0\0\0\x06\0\0\x09\x08\x02\x1f\0\x07\0\0\0\x0c\0\0\x09\x10\x03\xff\0\x03\0\0\0\x10\0\0\x09\x1c\0\0\0\0\0\0\0\0\0\0\0\0\x20\xa0\x20\0\x20\0\x43\x20\0\0\0\0\0\0\0\0\x02\x70\x10\x01\0\0\x28\0\x03\0\x70\0\0\0\x60\x10\x53\x6a\x28\x40\0\xa4\x03\0\0\x68\x6a\x94\0\x40\0\0\x04\xfc\0\x70\x28\xd6\x17\x51\x84\x28\x94\x66\x49\x9a\x60\x3c\x01\x4d\x7c\xfb\xff\xff\xff\xfc\x09\x76\x2a\xb1\0\x88\x08\0\0\0\x20\0\0\xf1\x3f\x3b\0\xac\0\xac\xc6\x18\x69\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\x58\x58\x56\x55\x54\x56\x54\x59\x57\x5a\x57\x59\x5b\x59\x5b\x56\x5e\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\x34\xd8\x20\0\x20\0\x1b\xbc\x3f\xa7\x20\0\x20\0\x15\x19\x39\x2b\x20\0\x20\0\x18\xcf\x1e\x03\x20\0\x20\0\x39\x9a\x3d\xed\x20\0\x20\0\x1c\x21\x33\x49\x20\0\x20\0\x1f\x5e\x2d\x94\x20\0\x20\0\x26\x2a\x37\x01\x20\0\x20\0\x19\xff\0\x01\x2a\xa2\x01\x31\xcd\xa2\0\0\0\x02\0\0\0\x02\0\x5b\x96\x1b\x82\x40\x9a\x90\0\0\0\0\x07\xff\xe0\x22\0\x11\x0c\x7a\x1f\x1f\x05\x0e\x01\xf4\x01\xf4\x01\xf4\0\0\0\x5d\0\x5e\0\x5f\0\0\0\x5a\0\x5a\0\x5b\0\x5b\0\x5c\0\x5c\0\x5d\0\0\xff\xd8\xff\xdb\0\x84\0\x08\x05\x06\x07\x06\x05\x08\x07\x06\x07\x09\x08\x08\x09\x0c\x14\x0d\x0c\x0b\x0b\x0c\x18\x11\x12\x0e\x14\x1d\x19\x1e\x1e\x1c\x19\x1c\x1b\x20\x24\x2e\x27\x20\x22\x2b\x22\x1b\x1c\x28\x36\x28\x2b\x2f\x31\x33\x34\x33\x1f\x26\x38\x3c\x38\x32\x3c\x2e\x32\x33\x31\x01\x08\x09\x09\x0c\x0a\x0c\x17\x0d\x0d\x17\x31\x21\x1c\x21\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\x31\xff\xc0\0\x11\x08\x01\xe0\x02\x80\x03\x01\x21\0\x02\x11\x01\x03\x11\x01\xff\xc4\x01\xa2\0\0\x01\x05\x01\x01\x01\x01\x01\x01\0\0\0\0\0\0\0\0\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x10\0\x02\x01\x03\x03\x02\x04\x03\x05\x05\x04\x04\0\0\x01\x7d\x01\x02\x03\0\x04\x11\x05\x12\x21\x31\x41\x06\x13\x51\x61\x07\x22\x71\x14\x32\x81\x91\xa1\x08\x23\x42\xb1\xc1\x15\x52\xd1\xf0\x24\x33\x62\x72\x82\x09\x0a\x16\x17\x18\x19\x1a\x25\x26\x27\x28\x29\x2a\x34\x35\x36\x37\x38\x39\x3a\x43\x44\x45\x46\x47\x48\x49\x4a\x53\x54\x55\x56\x57\x58\x59\x5a\x63\x64\x65\x66\x67\x68\x69\x6a\x73\x74\x75\x76\x77\x78\x79\x7a\x83\x84\x85\x86\x87\x88\x89\x8a\x92\x93\x94\x95\x96\x97\x98\x99\x9a\xa2\xa3\xa4\xa5\xa6\xa7\xa8\xa9\xaa\xb2\xb3\xb4\xb5\xb6\xb7\xb8\xb9\xba\xc2\xc3\xc4\xc5\xc6\xc7\xc8\xc9\xca\xd2\xd3\xd4\xd5\xd6\xd7\xd8\xd9\xda\xe1\xe2\xe3\xe4\xe5\xe6\xe7\xe8\xe9\xea\xf1\xf2\xf3\xf4\xf5\xf6\xf7\xf8\xf9\xfa\x01\0\x03\x01\x01\x01\x01\x01\x01\x01\x01\x01\0\0\0\0\0\0\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x11\0\x02\x01\x02\x04\x04\x03\x04\x07\x05\x04\x04\0\x01\x02\x77\0\x01\x02\x03\x11\x04\x05\x21\x31\x06\x12\x41\x51\x07\x61\x71\x13\x22\x32\x81\x08\x14\x42\x91\xa1\xb1\xc1\x09\x23\x33\x52\xf0\x15\x62\x72\xd1\x0a\x16\x24\x34\xe1\x25\xf1\x17\x18\x19\x1a\x26\x27\x28\x29\x2a\x35\x36\x37\x38\x39\x3a\x43\x44\x45\x46\x47\x48\x49\x4a\x53\x54\x55\x56\x57\x58\x59\x5a\x63\x64\x65\x66\x67\x68\x69\x6a\x73\x74\x75\x76\x77\x78\x79\x7a\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x92\x93\x94\x95\x96\x97\x98\x99\x9a\xa2\xa3\xa4\xa5\xa6\xa7\xa8\xa9\xaa\xb2\xb3\xb4\xb5\xb6\xb7\xb8\xb9\xba\xc2\xc3\xc4\xc5\xc6\xc7\xc8\xc9\xca\xd2\xd3\xd4\xd5\xd6\xd7\xd8\xd9\xda\xe2\xe3\xe4\xe5\xe6\xe7\xe8\xe9\xea\xf2\xf3\xf4\xf5\xf6\xf7\xf8\xf9\xfa\xff\xda\0\x0c\x03\x01\0\x02\x11\x03\x11\0\x3f\0\xf9\xfe\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x0e\xeb\xfe\x11\xed\x2f\xfe\x7d\x7f\xf2\x23\x7f\x8d\x2f\xfc\x23\xba\x57\xfc\xfa\xff\0\xe4\x46\xff\0\x1a\xe6\xe7\x91\xd1\xc9\x11\x7f\xe1\x1d\xd2\xbf\xe7\xd7\xff\0\x22\x37\xf8\xd2\xff\0\xc2\x3b\xa5\x7f\xcf\xa7\xfe\x44\x7f\xf1\xa3\x9e\x5d\xc7\xc9\x10\xff\0\x84\x73\x49\xff\0\x9f\x4f\xfc\x88\xff\0\xe3\x4a\x3c\x39\xa4\xff\0\xcf\xaf\xfe\x44\x7f\xf1\xa3\xda\x48\x39\x22\x1f\xf0\x8d\xe9\x3f\xf3\xe9\xff\0\x91\x1f\xfc\x69\x7f\xe1\x1b\xd2\x7f\xe7\xd3\xff\0\x22\x3f\xf8\xd1\xcf\x2e\xe1\xc9\x10\xff\0\x84\x6b\x49\xff\0\x9f\x4f\xfc\x88\xff\0\xe3\x4b\xff\0\x08\xd6\x93\xff\0\x3e\x9f\xf9\x11\xff\0\xc6\x8e\x79\x77\x0e\x48\x8b\xff\0\x08\xd6\x91\xff\0\x3e\x9f\xf9\x11\xff\0\xc6\x8f\xf8\x46\xb4\x9f\xf9\xf4\xff\0\xc8\x8f\xfe\x34\x7b\x49\x77\x0e\x48\x87\xfc\x23\x3a\x47\xfc\xfa\x7f\xe4\x47\xff\0\x1a\x5f\xf8\x46\x74\x8f\xf9\xf4\xff\0\xc8\x8f\xfe\x34\x7b\x49\x77\x0e\x48\x8b\xff\0\x08\xce\x91\xff\0\x3e\x7f\xf9\x11\xff\0\xc6\x8f\xf8\x46\x34\x8f\xf9\xf4\xff\0\xc8\x8f\xfe\x34\xbd\xa4\xbb\x87\x24\x43\xfe\x11\x8d\x23\xfe\x7c\xff\0\xf2\x23\xff\0\x8d\x2f\xfc\x23\x1a\x47\xfc\xf9\xff\0\xe4\x47\xff\0\x1a\x3d\xa4\xbb\x87\x24\x40\x78\x63\x48\xff\0\x9f\x3f\xfc\x88\xff\0\xe3\x4b\xff\0\x08\xbe\x8f\xff\0\x3e\x7f\xf9\x11\xff\0\xc6\x8f\x69\x2e\xe1\xc9\x10\xff\0\x84\x5f\x47\xff\0\x9f\x3f\xfc\x88\xff\0\xe3\x4b\xff\0\x08\xbe\x8f\xff\0\x3e\x7f\xf9\x15\xff\0\xc6\x8f\x69\x2e\xe1\xc9\x10\xff\0\x84\x5b\x47\xff\0\x9f\x3f\xfc\x8a\xff\0\xe3\x47\xfc\x22\xda\x3f\xfc\xf9\xff\0\xe4\x57\xff\0\x1a\x3d\xa4\xbb\x87\x24\x7b\x0b\xff\0\x08\xb6\x8f\xff\0\x3e\x7f\xf9\x15\xff\0\xc6\x8f\xf8\x45\xb4\x7f\xf9\xf3\xff\0\xc8\xaf\xfe\x34\x7b\x49\x77\x0e\x48\x87\xfc\x22\xda\x3f\xfc\xf9\xff\0\xe4\x57\xff\0\x1a\x3f\xe1\x15\xd1\xff\0\xe7\xcf\xff\0\x22\xbf\xf8\xd1\xed\x25\xdc\x39\x23\xd8\x5f\xf8\x45\x74\x7f\xf9\xf2\xff\0\xc8\xaf\xfe\x34\x7f\xc2\x29\xa3\xff\0\xcf\x9f\xfe\x45\x7f\xf1\xa5\xed\x25\xdc\x39\x22\x2f\xfc\x22\x9a\x3f\xfc\xf9\x7f\xe4\x57\xff\0\x1a\x3f\xe1\x14\xd1\xff\0\xe7\xcb\xff\0\x22\xbf\xf8\xd1\xed\x25\xdc\x39\x22\x1f\xf0\x8a\x68\xdf\xf3\xe5\xff\0\x91\x5f\xfc\x69\x7f\xe1\x14\xd1\xff\0\xe7\xcb\xff\0\x22\xbf\xf8\xd3\xf6\x92\xee\x1c\x91\x0f\xf8\x45\x34\x7f\xf9\xf2\xff\0\xc8\xaf\xfe\x34\x7f\xc2\x27\xa3\xe3\xfe\x3c\xbf\xf2\x2b\xff\0\x8d\x2f\x69\x2e\xe1\xc9\x10\xff\0\x84\x4f\x47\xff\0\x9f\x2f\xfc\x8a\xff\0\xe3\x47\xfc\x22\x7a\x3f\xfc\xf9\x1f\xfb\xfa\xff\0\xe3\x47\xb4\x97\x70\xe4\x88\x7f\xc2\x27\xa3\xff\0\xcf\x97\xfe\x45\x7f\xf1\xa5\xff\0\x84\x4b\x47\xff\0\x9f\x2f\xfc\x8a\xff\0\xe3\x47\xb4\x97\x70\xe4\x88\x7f\xc2\x23\xa4\x7f\xcf\x89\xff\0\xbf\xaf\xfe\x34\xbf\xf0\x88\xe9\x1f\xf3\xe2\x7f\xef\xeb\xff\0\x8d\x1e\xd2\x5d\xc3\x92\x22\x7f\xc2\x23\xa4\x7f\xcf\x89\xff\0\xbf\xaf\xfe\x34\x7f\xc2\x23\xa4\x7f\xcf\x97\xfe\x45\x7f\xf1\xa3\xda\x4b\xb8\x72\x44\x3f\xe1\x12\xd1\xff\0\xe7\xcb\xff\0\x22\xbf\xf8\xd1\xff\0\x08\x96\x91\xff\0\x3e\x5f\xf9\x15\xff\0\xc6\x8f\x69\x2e\xe1\xc9\x10\xff\0\x84\x47\x48\xff\0\x9f\x3f\xfc\x8a\xff\0\xe3\x47\xfc\x22\x5a\x47\xfc\xf9\x8f\xfb\xfa\xff\0\xe3\x47\xb4\x90\x72\x44\x4f\xf8\x44\xb4\x7e\xd6\x7f\xf9\x15\xff\0\xc6\x8f\xf8\x44\xb4\x8f\xf9\xf3\x1f\xf7\xf5\xff\0\xc6\x8f\x69\x2e\xe1\xc9\x11\x3f\xe1\x13\xd1\xff\0\xe7\xcc\x7f\xdf\xd7\xff\0\x1a\x4f\xf8\x45\x34\x7f\xf9\xf4\x1f\xf7\xf5\xff\0\xc6\x8f\x69\x20\xe4\x88\x7f\xc2\x2b\xa3\x7f\xcf\xa0\xff\0\xbf\xaf\xfe\x34\x7f\xc2\x2b\xa3\x7f\xcf\xa8\xff\0\xbf\xaf\xfe\x34\x7b\x49\x07\x24\x44\xff\0\x84\x5b\x46\xff\0\x9f\x51\xff\0\x7f\x5f\xfc\x69\x3f\xe1\x16\xd1\x87\xfc\xba\xff\0\xe4\x47\xff\0\x1a\x3d\xa4\x83\x92\x21\xff\0\x08\xbe\x8d\xff\0\x3e\xbf\xf9\x11\xff\0\xc6\x8f\xf8\x45\xf4\x5f\xf9\xf5\x3f\xf7\xf1\xff\0\xc6\x8f\x69\x2e\xe1\xc9\x11\x3f\xe1\x17\xd1\x71\xff\0\x1e\xa7\xfe\xfe\x3f\xf8\xd2\x7f\xc2\x31\xa2\xff\0\xcf\xaf\xfe\x44\x7f\xf1\xa3\xda\x48\x39\x22\x1f\xf0\x8c\xe8\xbf\xf3\xe8\x7f\xef\xe3\xff\0\x8d\x27\xfc\x23\x5a\x2f\xfc\xfa\x9f\xfb\xf8\xff\0\xe3\x4f\xda\x4b\xb8\x72\x44\x3f\xe1\x1b\xd1\x7f\xe7\xd0\xff\0\xdf\xd7\xff\0\x1a\x4f\xf8\x46\xf4\x6f\xf9\xf4\x3f\xf7\xf5\xff\0\xc6\x8e\x79\x77\x0e\x48\x89\xff\0\x08\xde\x8d\xff\0\x3e\x87\xfe\xfe\xbf\xf8\xd0\x3c\x39\xa3\x7f\xcf\x99\xff\0\xbf\xaf\xfe\x34\x73\xcb\xb8\x72\x44\x4f\xf8\x47\x34\x6f\xf9\xf4\x3f\xf7\xf5\xff\0\xc6\x8f\xf8\x47\x34\x7f\xf9\xf3\xff\0\xc8\xaf\xfe\x34\x7b\x49\x77\x17\x24\x40\xf8\x73\x47\x1f\xf2\xe7\xff\0\x91\x5f\xfc\x69\x0f\x87\x74\x71\xff\0\x2e\x7f\xf9\x15\xff\0\xc6\x8e\x79\x77\x0e\x48\x89\xff\0\x08\xf6\x91\xff\0\x3e\x7f\xf9\x11\xff\0\xc6\x83\xe1\xed\x23\xfe\x7c\xff\0\xf2\x2b\xff\0\x8d\x1c\xf2\xee\x3e\x48\xf6\x0f\xf8\x47\xb4\x8f\xf9\xf3\xff\0\xc8\x8f\xfe\x34\x9f\xf0\x8f\x69\x1f\xf3\xe7\xff\0\x91\x1f\xfc\x68\xe7\x97\x70\xe4\x88\x1f\x0f\x69\x1f\xf3\xe9\xff\0\x91\x1f\xfc\x68\xff\0\x84\x7b\x49\x3f\xf2\xe9\xff\0\x91\x1f\xfc\x68\xf6\x92\xee\x2e\x48\x89\xff\0\x08\xf6\x93\xff\0\x3e\x9f\xf9\x11\xff\0\xc6\x83\xe1\xed\x27\xfe\x7d\x3f\xf2\x23\xff\0\x8d\x1e\xd2\x41\xc9\x11\x0f\x87\xf4\x9e\xd6\xbf\xf9\x11\xff\0\xc6\x93\xfe\x11\xfd\x27\x3f\xf1\xeb\xff\0\x91\x1f\xfc\x69\xf3\xc8\x39\x22\x6c\x08\xf9\xea\x29\xc2\x1f\x71\x50\x58\xbe\x57\xb8\xa5\xf2\xbd\xc5\x21\x8b\xe4\xfb\x8a\x51\x17\xbd\0\x2f\x92\x3f\xbc\x29\xde\x48\xcf\x5a\0\x5f\x24\x7a\xd2\x88\x57\xd6\x80\x14\x42\xbe\xb4\xbe\x52\xfa\x9a\x04\x28\x89\x69\x7c\xb5\xa0\x62\x88\xd6\x8d\x8b\x48\x05\x08\x94\xbb\x12\x80\x17\x62\x76\xa3\x6a\xf6\x14\0\xbb\x53\xd2\x8d\xab\xe8\x28\0\x01\x7d\x29\x40\x5f\x4a\x04\x1f\x2f\xf7\x45\x2f\xca\x3b\x0a\x06\x1f\x2f\xa5\x1f\x2f\xa0\xa0\x05\xca\xfa\x0a\x01\xe7\xb6\x3e\x94\x58\x42\xe4\x7a\x0f\xca\x82\xc3\xdb\xf2\xa2\xc3\x10\x30\xa5\xdf\x40\x08\x5e\x8d\xfe\xf4\x08\x37\xd1\xbe\x80\x0f\x30\xd2\x79\x94\x0c\x37\xd1\xbc\xe6\x81\x08\x5c\xe6\x90\xb9\xa0\0\xb9\xa4\x2e\x7d\x68\x01\x0b\x1a\x42\xc7\xd6\x80\x10\x92\x4f\x5a\x4a\x60\x21\xa4\xa4\x02\x76\xe2\x92\x98\x09\x45\0\x21\xa4\xe8\x28\x01\x0d\x25\0\x25\x27\xb5\0\x04\x66\x92\x98\x05\x25\x20\x13\x14\x74\xa0\x02\x93\xad\x30\x12\x92\x90\x09\x8a\x4c\xf1\x4c\x02\x8f\xd2\x80\x13\xbd\x14\0\xd3\x49\x40\x16\x06\x69\xc0\x9a\x60\x2e\x69\x45\x20\x1c\x09\xa5\x07\x14\0\xa0\xd2\xe7\x8a\0\x50\x69\x73\x40\x0a\x0f\xa5\x2e\x68\0\x06\x97\x26\x90\x06\x68\xcd\0\x2e\x68\x07\x14\0\xbb\xa8\xcd\0\x2e\x68\xcd\0\x1b\xa8\xcd\0\x19\xa3\x3d\xe8\x01\x69\x73\x8a\0\x33\xc5\x04\xfa\xd0\x01\x9f\x5a\x09\xe6\x80\x0e\x94\x50\x01\x40\x1d\x68\0\x34\xbc\xd2\x01\x28\xa6\x01\x49\xf4\xa4\x02\x51\xf4\xa6\x01\x49\xde\x80\x0a\x4a\0\x28\xf5\xa0\x04\x3d\x69\x0f\x7e\xf4\0\x52\x1a\0\x43\x49\xf5\xa0\x02\x93\xaf\x5a\0\x43\x4d\xa6\x01\xc5\x21\xa0\x62\x1a\x4a\0\x0d\x27\xf2\xa4\x20\xa4\xa6\x02\x7b\x51\xde\x90\x08\x69\x31\x4c\x04\x22\x8a\0\x4f\xe5\x45\0\x21\xa4\xeb\xd2\x80\x10\xfb\x52\x50\x05\x8c\x52\x8e\x29\x80\xe0\x3d\x69\x45\x20\x16\x96\x80\x1c\x07\x34\xa2\x80\x0a\x5e\xf4\0\xa3\xa5\x02\x80\x16\x80\x68\x01\x45\x1f\x41\x48\0\xd2\xd0\x02\x83\x46\x68\x01\x45\x02\x80\x0f\xa5\x14\0\x0a\x5a\0\x05\x1d\x68\x01\x73\x8a\x28\0\xa2\x80\x0a\x5e\xf4\0\x63\xbd\x02\x80\x0c\xd2\x9a\0\x4a\x3a\xd0\x02\x50\x69\0\x74\xa2\x98\x09\x47\xd6\x80\x10\xd1\xde\x90\x09\x8a\x0d\x30\x0a\x6f\xd6\x80\x13\xad\x25\x03\x0e\xf4\x94\x08\x4e\xa6\x93\x19\xa0\0\x8c\x9a\x43\x40\x08\x45\x21\xe2\x80\x0c\x52\x50\0\x45\x25\x03\x12\x90\xd3\x10\x86\x8a\0\x4a\x28\x01\xa6\x83\xf4\xa0\x02\x93\xe9\x40\x08\x69\x28\x01\x29\x38\xc5\0\x59\xc5\x2e\x29\x80\xea\x51\x40\x0b\x40\xa4\x03\x85\x2f\xd6\x80\x16\x81\xed\x40\x0b\x4b\xd6\x80\x0a\x07\x4a\0\x28\xa4\x02\x8a\x5a\0\x3a\xd1\x9e\x68\0\xa7\x50\x02\x52\xe3\x9a\0\x5a\0\xa0\x05\xa0\x50\x01\x4b\x40\x09\x4a\x28\0\xa3\xa5\0\x14\x52\0\xfa\x9a\x3b\x53\x01\x69\x28\0\xef\x49\x40\x01\xa3\xeb\x40\x09\xc5\x14\0\x9d\xe8\xe3\x8a\x40\x2d\x37\x92\x29\x80\x52\x50\x02\x1e\x94\x94\0\x94\x1a\0\x4a\x4a\0\x4c\x52\x11\x40\x07\x7a\x4a\x06\x21\xa4\xef\x40\x01\xa4\xf6\xa0\x41\xf4\xa4\x34\xc0\x4a\x4a\0\x05\x25\0\x25\x1d\x0d\0\x21\xe6\x8c\x50\x02\x1a\x43\xc5\0\x34\xd2\x1a\0\xb5\xde\x94\x0a\0\x51\x4a\x29\x80\xea\x29\0\xa2\x96\x80\x0a\x70\xa0\0\x51\x40\x06\x0d\x14\0\x52\xe3\xbd\x20\x0e\xb4\xea\0\x28\xa0\x05\xa5\xa0\x03\x9a\x3d\xa8\x01\x4f\xd2\x8c\xd0\x02\xd2\x50\x02\xf7\xe6\x8c\x7a\xd0\x01\x4b\xd2\x80\x0a\x31\xcf\x5a\0\x5a\x28\x01\x29\x69\0\x86\x80\x29\x80\xb4\x98\xa0\x04\xa3\x14\x80\x28\xeb\xef\x40\x09\x8e\x28\xa0\x04\x34\x1f\x4c\xd0\x02\x13\x49\x40\x06\x39\xa4\xa0\x04\xfa\x52\x62\x98\x09\x49\x40\x05\x27\xd2\x80\x03\xf4\xa6\xd0\x02\x51\x40\x09\xd2\x92\x80\x0a\x42\x28\x01\x08\xa3\x14\0\x52\x1a\x60\x21\xa4\x14\x80\x28\xc5\x30\x13\x14\x94\0\xd3\x48\x78\xa0\x0b\x58\xa5\xed\x40\x0a\x39\xa5\x1c\x50\x02\xd2\xd0\0\x3a\x52\xe6\x80\0\x29\x68\0\xa5\x1d\x68\0\xa3\xb5\x20\x16\x96\x80\x0e\x94\xa2\x80\x0a\x5e\x94\0\x51\x40\x0b\x4b\xcd\0\x14\x0f\x6a\0\x28\xa0\x05\xcd\x14\0\x1e\x69\x68\0\xef\xcd\x2d\0\x25\x2e\x33\x40\x05\x14\x80\x31\x45\0\x14\x98\xa0\x03\x02\x8e\xd4\0\x52\x67\x9a\x60\x26\x68\xeb\xde\x90\x05\x25\0\x27\x7a\x4a\0\x4a\x0e\x7d\x29\x80\x84\x51\x8a\0\x4a\x43\xd6\x81\x89\x8f\x7a\x4a\0\x0d\x25\0\x25\x26\x28\x10\x94\x1c\xd0\x01\x49\x81\x40\x08\x47\x7a\x43\x40\x01\x14\x94\0\x87\xad\x25\x30\x0e\xd4\x1a\x06\x27\x7a\x69\xe9\x48\x42\x52\x53\x02\xd8\x14\xb4\xc0\x5e\xd4\x50\x02\x8a\x5a\x40\x14\xa2\x80\x14\x75\xa2\x80\x01\x4b\x40\x07\xbd\x03\xa5\0\x2d\x2f\x41\x40\x07\x7a\x5a\x40\x1d\xba\x52\x90\x08\xe6\x80\x01\xc6\x05\x2e\x38\xe6\x80\x0c\x73\x4b\x40\0\xa2\x80\x0a\x5c\x7a\xd0\x02\xd1\x40\x05\x02\x80\x0a\x5a\0\x3e\xb4\x50\x01\x8f\x4a\x0d\x20\x03\x45\x30\x0e\xb4\x52\0\x34\x94\xc0\x4a\x31\x40\x05\x27\x6e\xb4\0\x52\x50\x02\x52\x1a\0\x29\x28\x01\x28\x3e\xf4\x0c\x29\xa6\x80\x12\x8a\x04\x27\x7a\x42\x4d\x30\x12\x8a\0\x43\x45\0\x25\x25\x20\x13\x18\x14\x50\x02\x52\x66\x98\x09\x47\x1d\xa8\0\x3f\x5a\x4c\x50\x02\x1a\x4a\0\x4a\x69\xa0\x0b\x94\x75\xa0\x07\x71\xf8\xd1\x40\x06\x69\x45\0\x2d\x14\0\xbd\x28\xa0\x02\x97\xad\0\0\x51\x40\x0a\x3a\x51\x8a\x40\x28\xf5\xa5\xa0\x05\xa2\x80\x0e\xf4\xb4\0\x52\xf6\xa0\x03\xb7\x14\xbd\xe8\0\x14\x50\x01\xcd\x1f\x8d\0\x28\xeb\x8a\x28\x01\x40\xfc\xa8\xc5\0\x14\x50\x01\x4b\x8a\0\x0d\x14\x80\x4f\xc2\x8a\0\x29\x38\xa0\x02\x8a\x60\x25\x06\x90\x09\x8e\x29\x0d\0\x1d\x07\xd6\x90\xd0\x02\x52\x62\x80\x03\x49\xd2\x98\x09\x48\x78\x14\0\x1a\x42\x28\x18\x94\x94\x08\x3a\x52\x50\x01\x49\xf4\xa0\x04\x34\x1a\0\x69\x14\x62\x80\x12\x8c\x73\x4c\x04\xa4\xed\x40\xc4\x3d\x68\xa0\x42\x7d\x69\x0d\0\x34\xd2\x50\x05\xca\x51\x4c\x05\xa0\x52\x01\x45\x03\xeb\x40\x0a\x38\x14\xb4\0\x7f\x2a\x28\x01\x68\xfa\xf5\xa0\x02\x97\xad\0\x14\x0a\x40\x3a\x8a\0\x29\x68\0\xeb\x4b\x40\x0a\x3f\x5a\x33\x8a\0\x31\xf8\x9a\x5a\0\x28\xfa\x50\x02\xf3\x46\x33\x40\x05\x2d\0\x14\x50\x01\xc5\x2f\x6a\x40\x02\x8a\x60\x14\x76\xa4\x01\x45\0\x25\x27\x5e\x94\0\x1a\x0d\0\x25\x26\x68\0\xef\x45\0\x34\xff\0\x2a\x3a\x74\xa6\x02\x51\x40\x09\x9a\x43\x48\x04\xfc\x69\x0d\x31\x86\x29\x3b\xd0\0\x79\xa4\xc0\xa0\x04\xc5\x06\x81\x0d\xe2\x83\x40\xc4\xcd\x06\x81\x08\x69\x0f\x3d\x28\0\x3e\x94\x82\x80\x10\xd2\x53\0\xa4\x3e\xb4\x86\x25\x21\xe9\x4c\x42\x1a\x43\xd2\x80\x2d\x8a\x51\x4c\x05\xa5\xa4\x01\x47\x51\x40\x0b\x4b\x40\0\xa2\x80\x16\x81\x40\x05\x28\x34\0\x51\x48\x05\x1d\x29\x68\0\xa5\xcd\0\x02\x94\x50\x02\x93\x40\xeb\x40\x05\x14\0\xb4\x50\x01\x4b\xd6\x80\x01\x4b\x40\x07\xe1\x47\x6a\0\x31\x45\0\x2f\x53\x45\0\x02\x8a\x40\x14\x87\xd6\x80\x0a\x28\x01\x31\x40\x14\xc0\x4a\x3d\xa9\0\x66\x9b\x40\x01\xa4\x34\0\x9d\xa8\xa6\x02\x52\x50\x31\x38\xa4\xa0\x02\x93\x34\0\x52\x1a\0\x4a\x43\x40\x80\xd2\x50\x31\x0f\x4a\x4f\x5a\x04\x06\x92\x81\x85\x25\0\x25\x06\x98\x0d\xa4\x34\0\x7d\x7a\xd2\x50\x21\x0d\x21\xa0\x0b\x78\xcd\x3b\xe9\x4c\x02\x97\xa5\x20\x0e\xf4\xb4\0\x52\xf5\xa0\x03\xa5\x14\0\xbf\x4a\x0f\x26\x80\x0a\x5a\0\x3b\x51\x48\x05\xc5\x14\0\xb4\x63\xdf\x8a\0\x5a\x51\x40\0\xe9\x4b\x40\x07\x4a\x5a\0\x07\x7a\x3b\xd0\x02\x8e\x28\xa0\x03\x81\x4b\x40\0\xa2\x80\x16\x93\xb5\0\x2d\x14\0\x51\xd6\x80\x0e\x94\x94\0\x52\x76\xa4\x01\xde\x83\xd2\x98\x08\x4d\x25\0\x1f\x51\x49\x40\x09\x49\x40\x01\xa0\xd0\x02\x52\x50\x02\x1f\x7a\x3f\x95\0\x25\x21\xa0\x62\x51\x40\x84\x34\x98\xcd\x03\x03\x49\xf4\xa0\x04\xa2\x81\x08\x68\xa6\x31\x29\x28\x01\x0d\x21\xa0\x04\x34\x1a\x04\x25\x21\xa0\x06\x93\xcd\x21\xa0\x0b\xa2\x94\x53\x01\x45\x18\xa4\x02\xd1\xde\x80\x01\xd2\x96\x80\x01\x45\0\x14\xb4\0\x52\x8a\0\x31\xcd\x02\x80\x17\x34\x7e\x34\x80\x5a\x3a\x50\x02\x8e\x4d\x1f\x4a\0\x5f\xad\x2d\0\x14\x1a\0\x3b\x52\xd0\0\x3b\x51\x9a\0\x5a\x5a\0\x53\x49\x40\0\x14\xb4\0\x51\xd6\x80\x0a\x29\0\x67\x9f\x7a\x4c\x50\0\x71\x48\x45\x30\x0a\x41\x40\x07\x02\x93\xad\0\x1e\xf9\xa4\x26\x80\x13\x34\x94\0\x94\x50\x31\x05\x14\0\x86\x92\x81\x05\x14\0\x94\x9f\x5a\x06\x21\xa2\x80\x10\xfb\xd2\x62\x80\x0a\x4a\x62\x10\xd0\x68\x18\xd3\x47\xd2\x80\x10\xd2\x76\xa0\x02\x90\xd0\x21\x3e\xb4\xd3\x40\x09\x48\x68\x02\xed\x28\xa6\x02\xd2\xd2\0\xa2\x80\x17\xb5\x14\0\x77\xa3\x34\0\x51\xd2\x80\x0c\xd2\xd0\x02\xf6\xa3\x34\0\x01\x4a\x28\0\x1c\x52\xfd\x68\0\x14\xa3\xda\x90\x05\x2e\x71\x4c\x02\x97\xeb\x48\x03\xeb\x4b\x40\x07\xd2\x8c\x50\x02\x8a\x5a\0\x28\xc7\x34\0\x76\xa3\x8a\0\x5a\x4a\0\x5c\xd1\xef\x40\x09\x45\0\x14\x86\x80\x0e\x29\x0d\0\x26\x68\x26\x80\x1b\x9a\x0f\xbd\x20\x13\xeb\x46\x79\xa6\x02\x73\x8a\x4c\xd0\x01\x9a\x28\x01\x0d\x21\xfa\x66\x80\x12\x82\x68\0\xa4\x34\0\x94\x9f\xa5\x03\x16\x93\xf1\xa0\x42\x52\x50\x02\x66\x92\x98\x09\xd6\x83\xf5\xa4\x03\x68\x34\xc6\x25\x21\x34\x08\x29\xb9\xa0\x04\x34\x94\x01\x78\x51\x4c\x05\xa2\x80\x0f\xa5\x28\xa4\x01\x9a\x28\x01\x7b\xd1\x40\x09\x4b\x9a\0\x29\x68\x01\x69\x0f\x5a\0\x05\x2e\x48\xa0\x03\x39\xa5\xa0\x03\x34\xb4\0\xa3\x81\x47\x4e\x94\0\x03\x4b\xf8\x52\0\xa5\x14\x08\x29\x45\x03\x17\x34\x03\xc5\0\x19\xe6\x8c\xd0\0\x08\xfa\xd0\x0e\x7d\x28\x01\x73\xef\x48\x58\x0e\xa4\x50\x02\xe4\x62\xa1\x7b\xbb\x75\x24\x19\xa3\x04\x71\x8d\xc2\x81\x12\xa9\x04\x64\x10\x47\xb5\x19\xa0\x60\x4e\x29\x91\x4c\x93\x20\x78\x9c\x3a\x9e\xe0\xe4\x50\x04\x69\x75\x03\xdc\x3c\x0b\x22\x99\x50\x65\x90\x1e\x40\xa4\x96\xee\x08\xa7\x48\x5e\x45\x59\x24\xfb\xaa\x4f\x26\x8b\x31\x5c\x44\xbb\x82\x44\x0e\x92\xa9\x52\xdb\x33\x9e\xfe\x95\x26\x7f\x3a\x06\x44\x2e\x62\x37\x26\xdc\x37\xef\x02\xee\x2b\xed\x52\x50\x02\x66\x80\x78\xa0\x03\x3e\x94\xd7\x7d\x88\x58\xf4\x03\x3c\x50\x06\x5b\xeb\xf6\xaa\xea\x9e\x4d\xc1\x66\xe8\x3c\xa3\x93\x56\x2d\xb5\x38\x6e\x26\x48\x95\x64\x47\x91\x4b\x61\xd7\x1c\x03\x8a\xae\x52\x54\x89\xae\x6e\x56\x04\xde\x55\xdf\x90\x30\x83\x27\x9a\x90\xb7\x1d\x69\x58\xa0\xdc\x29\x37\x02\x71\x40\x08\x5c\x6e\xc7\x7a\x18\xe0\x66\x80\x0c\xd2\x9f\xc2\x90\x0d\x66\0\x1c\x9e\x29\xbb\xfb\x90\x71\x4c\x05\xcf\x14\xc2\xf8\x34\0\x33\x63\xa7\x34\x85\xb8\xcd\0\x26\xf1\xd7\xd2\x9a\x1f\x3d\xbf\x1a\0\x52\x71\x48\x1b\x23\x39\xa0\0\x36\x49\xf6\xa4\xcd\0\x26\x7d\x69\x09\xe6\x80\x10\x91\x8a\x32\x0f\x4a\x60\x5d\xa5\xcd\0\x19\xa0\x75\xa4\x02\xd0\x38\xa0\x05\x34\x50\x01\x47\x38\x3e\xb4\x01\x8d\x75\x7d\xa9\xc3\x3c\x31\x79\x30\x6e\x98\x90\xbf\x31\x23\x81\x93\x52\x7f\x68\xcd\x6d\x3c\x31\xea\x1e\x54\x7b\xd1\x99\x8a\x9e\x01\xcf\x02\xaa\xc8\x9b\xbe\xa5\x99\x6f\x91\xe0\x8a\x4b\x69\x62\x65\x69\x15\x49\x63\xf9\x81\xef\x4e\x97\x52\xb4\x86\x61\x0c\x93\xa0\x93\xd3\x3d\x29\x58\x77\x2c\x4f\x21\x4b\x79\x1d\x59\x41\x54\x2c\x0b\x74\x1c\x77\xaa\x63\x54\xb7\x8a\x18\xbe\xd3\x32\xef\x74\x0f\xf2\x82\x77\x7b\x8a\x12\x06\xec\x2a\x6b\x36\x2c\xca\x16\x70\x77\x60\x74\x38\xc9\xf5\x3d\xa9\xd7\x7a\x9d\xb5\xb4\x9e\x5b\xc8\x77\x80\x09\x01\x49\xc0\xf7\xc5\x16\x62\xe6\x42\x4f\xa9\x5b\x44\x91\xb1\x94\x9f\x37\x94\x0a\x0b\x16\x1f\x41\x49\x16\xab\x6c\xf6\xd2\xcc\xb3\x7c\x91\x9d\xac\x48\x23\x1f\xe7\x34\x59\x85\xd0\xb6\x9a\x94\x17\x13\x08\xd1\x9d\x64\xc6\x76\x3a\x90\x48\xf5\xa9\x2f\xae\x0c\x37\x76\x89\xe6\xec\x12\x16\x05\x76\xe7\x76\x07\xaf\x6a\x2d\xa8\x5f\x42\x05\xd6\xad\xde\x54\x5d\x92\x6c\x66\xd8\xb2\xec\x3b\x18\xfa\x66\xae\x4f\x70\x90\x40\xd3\x4e\xdb\x23\x4e\x49\xa1\xa0\xb9\x5a\x1d\x66\x17\x90\xa4\xd1\x4b\x01\xdb\xbc\x79\xa3\x19\x1d\xe9\x21\xd6\x62\x92\x58\xc3\x41\x34\x71\xca\x71\x1c\x8e\xb8\x56\x3d\xbf\x3a\x39\x58\x73\x11\xbe\xba\x81\xe5\xf2\xad\xa7\x91\x20\x25\x64\x60\x06\x17\x07\xad\x5a\x87\x54\x8a\x59\xda\x25\x07\x84\x12\x06\x3d\x19\x4f\x71\x47\x28\x5c\xa8\xda\xf0\x2b\x17\x95\x6d\x24\x8d\x3e\xef\x2d\x46\x39\xc1\xc7\xe1\x4b\x1e\xa4\xb6\xcd\x76\xf2\x79\xd2\x38\x94\x22\xa1\x39\xcb\x11\xc0\x5f\x4a\x39\x45\xcc\x39\x75\x69\x5a\xe1\xad\x6e\x2d\x1e\xde\x43\x11\x70\x77\x06\xfe\x55\x53\x4e\xd5\xa7\x86\xc6\xda\x49\xed\xdd\xa1\x62\x10\xcc\x5f\x24\x92\x71\x9c\x7a\x66\x9f\x28\x5c\xb5\xa6\x5e\x5d\x4b\xab\xde\x47\x24\x78\x89\x36\xe3\xe7\xce\xde\x3f\xad\x4d\xa8\x41\x35\xc5\xda\x87\x98\xc5\x66\x8a\x59\xf6\xbe\xd2\xc7\xb0\xcf\xa5\x2b\x59\x8f\x74\x63\xa5\xec\xc9\x6d\x2d\xbd\xbc\xf2\x34\x13\x5c\xac\x31\x4c\x4f\x21\x4f\x5c\x1a\xb5\x3b\xbe\x91\x79\xb6\x19\x1d\xa2\x92\xdd\xdc\x2b\xb1\x6c\x32\xf3\x9e\x69\xd8\x92\x23\x1b\x59\xd8\xda\x6a\x4b\x24\x8d\x70\xee\x86\x56\x2f\xf7\xd5\xbb\x62\x99\x7c\x23\x97\x51\xbc\x96\xea\xce\xe2\x78\xd3\x6e\x0a\x1c\x05\0\x73\xdf\x9a\x60\x74\x3a\x7c\x91\x4b\x67\x13\x5b\x9f\xdd\x6d\x1b\x7e\x95\xcc\xda\x35\x9f\x9b\x7a\x97\x56\x12\x5c\x49\xe7\xbe\x19\x23\xdd\xc7\xa6\x69\x21\xbb\x68\x5c\xb5\x92\xf2\xc3\x43\x85\x37\x25\xb9\x69\x0e\x5e\x53\xfe\xad\x73\x91\xc7\x73\xed\x4d\x4d\x66\xe0\xda\x5f\x95\x9a\x39\x9e\xd9\x54\xa4\xca\x98\x0d\x93\xe9\x47\x2d\xc2\xf6\x34\xaf\x6e\xe5\x8a\xea\xc5\x11\x80\x49\xc9\x0f\xc7\xb7\x15\x17\x86\x49\x1a\x4a\xf3\xcf\x98\xe0\x7f\xdf\x46\x95\xb4\x1f\x53\x2e\x42\xd6\xde\x22\xb8\xbd\x53\xf2\x24\x89\x1c\x83\xfd\x96\x5e\xbf\x9e\x29\x2e\x0b\x5d\x6b\x91\x5e\x06\xf9\x12\x71\x02\x7e\0\x93\xfa\xd5\x13\xe4\x32\xe1\xa7\xb9\xb5\xb7\x26\x6d\xa5\x6f\xb6\x0d\xaa\x3a\xe4\xe0\xfe\x15\xa0\x0d\xe5\xfc\xf7\x3e\x4d\xdb\x42\x2d\xdb\xcb\x4c\x28\xf9\x98\x0e\x49\xe3\xa5\x21\xea\x56\x10\xdd\x4f\xac\x88\xe6\x9c\xc3\x27\xd9\x86\xf6\x8b\x1c\xfc\xdd\x8d\x24\x5a\x85\xd7\x97\x04\x6d\x2b\x12\x97\x9e\x4b\x31\xea\xeb\xef\x4e\xc8\x5a\xa2\x6b\xfb\xa9\xa3\xd4\x6e\x23\x49\x58\x28\xb3\x67\x03\xb0\x6c\xf5\xaa\xc2\x3b\xc4\x87\x4f\x94\xdf\xcc\x5e\xe9\x82\x37\x23\0\x11\x9e\x3d\xfd\xe9\x59\x0d\xdc\xbd\xa6\x99\x61\xd4\x6e\xec\xda\x69\x25\x8e\x30\xac\xa5\xce\x48\xcf\x5e\x6b\x4b\x9c\x13\xba\x93\x29\x19\x77\xc4\x7f\xc2\x43\xa7\x7f\xbb\x27\xf2\xa8\xb5\x08\x05\xce\xbb\x6f\x19\x76\x55\x31\x36\x76\x9c\x12\x33\xd2\x99\x2f\xa9\x46\xe0\xcb\x0d\xbd\xe4\x2b\x33\x95\x82\x74\x09\x96\x24\x80\x48\xe3\x34\xfb\xe4\x90\x5c\x5c\x4b\x3a\xcb\x3a\x29\xf9\x24\x82\x4e\x62\xc7\x6d\xb9\xea\x29\x88\xdd\xb3\x70\xf6\x91\x3a\xb9\x90\x30\x1f\x31\xea\x7d\xe9\xe0\x61\x8b\x13\xde\xa4\xb0\x1c\xb8\x3d\x73\x4f\x93\xee\xf3\x48\x62\x29\xe4\xd3\x33\xf2\x81\x40\x0b\x80\x72\x29\x1b\x21\x79\xa0\x07\x74\x15\x18\xf9\x86\x73\x40\x07\x73\x4d\xee\x3e\xb4\0\x60\x12\x7d\x31\x48\x4e\x30\x0d\0\x2b\x63\x69\xcd\x33\xab\x28\xf6\xa6\x03\x4f\x1b\x85\x2f\x1c\x11\xde\x81\x08\x46\x54\xd3\x79\x18\x3e\xa2\x80\x10\x8c\x05\xc8\xe2\x8e\xed\x8a\0\xd1\xa2\x81\x87\xd2\x94\x50\x02\xd1\xcd\x20\x0e\x3b\x52\xe6\x80\x0a\x33\x40\x19\x7a\xae\x7f\xb4\xf4\xcf\xf7\xdf\xff\0\x41\xa4\xbc\x89\x64\xd7\x6d\x0b\x28\x6d\xb1\xb1\xe7\xb7\x3f\xfd\x7a\xa4\x49\x42\x64\x09\x24\xe1\x46\0\xbf\x8c\x80\x07\x4e\x94\xdb\x99\x16\x16\xbb\x31\xca\xb9\x67\xcb\x5a\xcc\x99\x2e\xdc\x74\x23\xf4\xaa\x24\xdc\xb8\x25\xb4\xa9\x77\x26\xc2\x61\x6c\xaf\xa7\xcb\x59\x7a\x6a\x13\x7f\x60\x59\x73\x8b\x11\xf8\x73\x52\x8a\x7b\x91\x79\x7b\x7c\x3d\x7a\xa9\x1f\xcd\xe7\x1c\x60\x7f\xb6\x29\xba\x84\xcf\xf6\xbb\x94\x91\x9a\x02\x50\x05\x44\x8f\x26\x5e\x3a\xe6\x99\x22\x69\x6f\xe4\xb5\x9d\xcb\xc6\xef\x08\x80\xc5\x95\x52\x76\xb0\x6a\x67\x95\x25\xdd\xa5\xeb\x45\x13\xa9\x17\x62\x42\x8a\x3e\x6c\x63\xf9\xf3\x9a\x7d\x44\x5c\xd3\xe1\x8e\x6d\x52\x29\x22\x7b\x99\xc4\x6a\x49\x92\x4e\0\xcf\x60\x30\x2a\xf6\xad\x13\xc9\x7d\xa7\xb2\x21\x60\x8e\xfb\x88\x1c\x0f\x97\xbd\x4f\x52\x96\xc6\x4c\x6b\x2c\x52\xa2\xd9\x45\x75\x1c\xa2\x5f\x9e\x16\x1b\xa3\x51\x9e\x79\xad\x8d\x66\xde\x5b\x8d\x32\x44\x89\x77\x3e\x43\x6d\xf5\xc1\xe9\x4d\xbd\x41\x2d\x19\x52\x57\xb8\xd5\x11\xe1\x8a\xd1\xe2\x5f\x29\xb2\xf2\xae\xd2\x1b\xd0\x53\x08\x9e\xf2\x1b\x5b\x3f\xb1\xc9\x13\x45\x22\xb3\xbb\xae\x15\x42\xfa\x1e\xf4\x06\xac\x8a\x09\xa7\x8c\x6a\x50\xc1\x69\x24\xbe\x64\xf2\x2a\xb2\x81\x8c\x9e\x39\xf4\xa5\xd5\x2c\xa6\xb3\xb0\xb3\xf2\x58\x7d\xa0\xaf\xd9\x8f\xb8\x61\xfd\x08\xa2\xe2\xd6\xc6\x80\xb0\x78\xef\x6c\x4c\x51\xfe\xea\x08\xd9\x49\xcf\x4e\x2a\xa5\xed\x94\xe2\x4b\x89\x8e\xd5\xc5\xca\x4b\x16\xf6\0\x39\x03\x18\xcd\x24\xc6\xd0\x91\xbd\xcd\xd6\xbd\x1f\xda\x21\x10\x91\x6e\xc0\x26\xe0\xc4\x73\xd4\xd5\x8f\xec\xeb\x93\xa0\x43\x6d\xb0\x79\xa8\x54\x95\xcf\xa3\x66\x8b\xa4\x16\x6c\x9a\xd2\xd6\xea\xdf\x57\xb8\x7f\x28\x18\x27\x0a\x7c\xcd\xdd\x08\x18\xc6\x29\x35\x9b\x4b\xdb\xab\x98\xbc\xb8\xd2\x5b\x74\x19\x68\xcb\xed\xdc\xdd\xb3\x4a\xea\xe3\xb3\xb0\xd9\xac\xaf\x6e\xec\xbc\xa7\x82\x1b\x67\x81\xd5\xe1\xda\xd9\x52\x45\x2a\x69\xf7\x57\xb7\x06\x6b\xf5\x48\x94\x42\x62\x54\x56\xdd\xd7\xa9\xa2\xe8\x2c\xc8\xa2\xd2\xf5\x09\x22\x82\xd2\xe9\xa2\xfb\x3c\x0e\x1b\x78\x27\x73\x81\xd0\x62\xa7\x9e\xd3\x55\x59\x26\x48\x24\x85\xe2\x94\x9d\xad\x21\x39\x8f\x3d\xbd\xe8\xba\x15\x99\x7f\x4e\xb3\x16\x56\x91\xc0\xad\xbb\x60\xc6\x7d\x6a\x1d\x32\xc6\x4b\x3f\xb4\x97\x65\x63\x34\xcd\x20\xc7\x60\x69\x5c\xab\x0c\xd5\x74\xf9\x2e\x5e\xde\x5b\x76\x8f\xcc\x81\x8b\x05\x90\x65\x5b\x35\x51\x74\x5b\xa9\x0d\xd1\xba\x9d\x0f\xda\xa3\x01\xb6\xae\x36\xb0\xe9\x8f\x6a\x69\x8a\xc3\xc6\x95\x79\x24\xf6\xd3\x5d\xdd\xa3\x7d\x98\xfc\xaa\x89\x80\x78\xc6\x7e\xb5\x6f\x4b\xb2\x36\x76\x82\x1d\xfb\xb0\xc5\xb3\x8c\x75\x39\xa1\xb0\x48\x82\x5d\x1d\x65\x6b\xed\xf2\x71\x76\x14\x63\x1f\x74\x81\xd7\xde\x92\x0d\x1d\x62\xb6\xb4\x88\x4a\x49\xb7\x97\xcc\xdc\x47\x2e\x79\xff\0\x1a\x39\x82\xc3\x64\xd1\x41\xb5\x31\x24\xe5\x1b\xcf\xf3\x95\xf6\xf4\x39\xcf\x4a\x6c\xba\x3c\xbe\x64\x8f\x05\xe3\x44\x26\0\x4a\x02\x83\xb8\xe3\x04\x8f\x4a\x2e\x16\x25\xb5\xd2\x62\xb5\xb9\x59\x63\x76\xf9\x63\x11\x85\x3c\xf1\x9c\xd4\x52\xe8\xb1\xb4\x72\x2a\xca\xe8\xed\x31\x99\x58\x7f\x0b\x51\xcc\x1c\xa2\x45\xa2\x28\x92\x59\x66\xb8\x96\x59\x25\x88\xc6\xc4\xe3\xa1\xa9\xe4\xd3\x63\x78\x6d\x63\xde\xe0\x5a\x95\x64\x39\xeb\x81\x8e\x68\xe6\x0e\x50\x7d\x3a\x36\x96\x79\x44\x92\x2b\xce\xa1\x58\xa9\xc1\x18\xee\x2a\xc4\x71\x6c\x8c\x2e\x4b\x63\xb9\xea\x69\x5c\x76\x21\x9a\xce\x39\xae\x61\xb8\x24\x87\x83\x3b\x70\x78\xe4\x50\xd6\x71\x9b\xa4\xb9\x39\xde\x8a\x54\x73\xeb\x4e\xe1\x62\x29\x74\xbb\x69\x3c\xe0\xc1\xbf\x7c\xc1\xdf\x9e\xe3\xa5\x47\x36\x8d\x6f\x34\xcf\x21\x69\x57\xcc\x39\x75\x57\x20\x37\xd4\x51\xcc\x1c\xa8\xba\x90\xa4\x71\x2c\x68\xb8\x55\x18\0\x7a\x52\xed\x18\x22\x90\xc0\x28\x18\xc7\x6a\x08\xc8\xe9\x40\x08\x54\x66\x9a\x50\x71\x40\x0a\x06\x0e\x45\x1d\x45\0\x21\xa6\x6d\x14\0\x15\x04\xd2\x6d\x18\xc7\x6a\0\x02\x80\x29\x36\x8e\xb4\xee\x02\x37\x23\x9a\x4c\x62\x90\x08\x54\x60\xfb\xd3\x42\xf2\x3d\xbb\xd3\x10\xe3\x4d\x23\x23\x14\x0c\x4c\x64\x60\x8c\xd2\x05\0\xe6\x81\x17\xa9\x68\x18\x51\x40\x0b\x4a\x28\x01\x78\xa4\xfc\x29\0\x51\x40\x08\x42\xb1\x19\0\x91\xd3\xda\x97\0\x9c\xe0\x64\x77\xa0\x04\xda\xa7\xb7\x5e\xb4\x18\xd4\x9d\xc5\x41\x6f\x5c\x73\x40\x0e\x03\xd2\x80\xbc\xf4\x14\0\xb8\xa3\x03\xb8\xfd\x28\x02\xad\xdd\x8f\x9e\xe8\xf1\xcd\x24\x0e\xbc\x65\x0f\x04\x7b\x8a\x7d\x8d\x94\x76\x71\xb2\x21\x66\x2c\xc5\x99\x98\xe4\xb1\xf5\x34\xef\xa0\xad\xa9\x64\x71\x4d\x79\x51\x1d\x11\xdd\x43\x3e\x76\x82\x79\x38\xeb\x48\x07\x8e\x69\x78\xa0\x03\xa5\x2d\0\x47\x05\xb4\x50\x19\x0c\x4b\x8f\x31\xcb\xb7\x3d\x49\xeb\x55\x2e\x5f\x4f\x17\x6f\x71\x39\xfd\xed\xa2\x8d\xc4\xe4\xec\x07\xa7\x14\x03\xb1\xa0\x08\x38\x35\x0d\xd2\xdb\x5c\x0f\xb2\xdc\x6c\x72\xe3\x3b\x0f\x52\x3d\x68\x02\x08\xed\xec\xb4\x98\x5e\x68\xe3\xd8\x0e\x03\x30\xcb\x13\xcf\x15\x66\x2b\xa8\xa4\xb8\x96\x15\x3f\x3c\x58\xdc\x08\xe9\x91\x91\x43\xd4\x36\xd0\x9b\x75\x47\x73\x71\x15\xac\x2d\x34\xcd\xb1\x17\x19\x34\x80\x94\x30\x20\x1c\xf1\x46\x47\x5a\x06\x41\x0d\xec\x72\xde\xcd\x6c\xa0\xef\x84\x29\x63\xd8\xe7\xd2\xac\x64\x67\x19\xa2\xc2\x41\x91\x8c\xe6\xa9\x0d\x4a\x35\x8e\x16\x9e\x36\x88\xcd\x21\x45\x53\xd8\xf3\xd7\xf2\xa2\xc1\x71\x6d\x75\x18\xa6\x79\xd5\xb1\x1f\x93\x29\x8b\xe6\x3f\x78\xf1\xfe\x35\x6f\xb5\x16\x03\x15\x75\x7b\xd9\xde\x7f\xb2\xe9\xde\x6c\x71\x48\xc9\xbb\xcd\x03\x24\x7b\x51\x36\xb8\xc6\xca\xde\x6b\x6b\x7d\xf2\x4d\x27\x95\xb1\x9b\x6e\xd6\xf7\xaa\xe5\x26\xe4\xf6\xd7\x3a\x94\x93\x28\xb8\xb1\x48\xa2\x27\x96\x12\x82\x45\x4b\xa6\x5e\x9b\xd4\x98\x94\x0b\xe5\x4a\xd1\xf0\x7a\xe3\xbd\x26\xbb\x15\x77\xd4\xb6\x78\x06\xb1\xae\xb5\x79\x53\x4e\xb8\xbc\x89\x15\x95\x24\x09\x16\x7f\x8b\x90\x09\xfc\xf3\x42\x57\x06\xec\x36\x5d\x43\x51\xb2\xf2\xe4\xbd\xb7\x80\xc2\xcc\x15\x9a\x36\x39\x5c\xfd\x6a\xd5\xa5\xeb\x4d\x7f\x77\x6e\xe1\x42\xc0\x57\x69\xee\x72\x33\x4e\xcb\xa0\x93\x7d\x48\x27\xd4\xe5\x49\xf5\x08\xd5\x53\x16\xd0\xf9\x8b\xc7\x53\x8c\xf3\x55\x9a\xff\0\x54\x8e\xc1\x2f\x5d\x6d\x5e\x2d\xa1\xca\x8c\x86\xc5\x16\x41\x76\x6c\x45\x28\x96\xdd\x65\x40\x70\xca\x18\x7e\x55\x91\xa6\xdc\xea\xb7\xf6\x89\x70\xb3\x5b\x22\xb6\x46\x0a\x1c\xf0\x71\xeb\x42\xb7\x51\xbb\xf4\x2c\x5f\xdc\x5e\x59\x5b\x43\x39\xd9\x22\xc6\x7f\x7e\x15\x7a\x8f\x51\xf4\xa4\xb6\xbe\x7b\xfb\xf2\x2d\x18\x1b\x48\x97\xe7\x7c\x7d\xe6\x3d\x85\x16\x5b\x8a\xee\xf6\x29\x4d\xaa\x5d\xff\0\x67\x3c\xaa\xea\xb2\x0b\xb3\x08\x3b\x73\x81\x9a\xbd\x6f\x0d\xf0\x99\x4c\xba\x82\x3a\x83\x92\x82\x20\x32\x29\xd9\x02\xbb\x34\x71\x41\xf4\xa8\x28\x29\x0d\0\x44\x58\xe3\xf1\xc5\x04\x91\xc9\x3d\xe9\x80\x87\x24\xfa\x0a\x01\xc3\x6d\x27\x3e\xf4\0\xac\x70\x39\xa6\x86\xcd\0\x30\xb7\xa5\x28\x39\x24\x11\x40\x03\x36\x0f\x4a\x6b\x30\xf4\xa0\x06\x96\xe4\x11\x46\xfe\x32\x78\xa6\0\x1b\x9c\x74\xcd\x1b\xbe\x5a\0\x4e\x69\x33\xc6\x4d\x02\x10\x9c\x1e\x3a\xd2\x6e\xe0\xe6\x80\x2f\x51\xd6\x81\x8b\x45\0\x2d\x2e\x68\0\xef\x46\x69\0\x0c\xd2\xd0\x06\x1f\xdb\xde\xde\x16\x36\xf0\xe5\x9a\xe9\xa2\xc1\x6e\xa7\xd6\x9c\x97\xda\x89\x9a\x5b\x71\x0c\x1e\x6c\x20\x33\x31\x63\xb7\x04\x71\x8f\x7a\xbb\x11\x77\xd0\x7b\xea\x53\xc9\x69\x6f\x2d\xba\x45\x1f\x9a\x32\xcf\x33\x61\x57\xdb\xde\x9b\x16\xb1\x23\xda\x12\xa8\x8d\x3f\x9d\xe4\xae\x0f\xca\x4f\xaf\xd2\x95\x83\x98\x55\xd4\xae\xed\xef\x1e\x1b\xd4\x8c\x2c\x70\xb4\xbb\x93\x3c\xe0\xd4\x51\x6b\x72\x89\x21\x69\x1e\xd9\xe3\x95\x82\x94\x8d\xb2\xc9\x9f\xe7\x4e\xd7\x0e\x66\x69\xea\xf7\x52\x5a\x69\xf2\x5c\x44\xa1\x9d\x71\x80\x7e\xa2\xa8\xbc\xfa\x97\xdb\x63\xb4\xf3\x20\x0d\x2c\x7e\x66\xf0\xa7\xe5\x19\xe4\x63\x3c\xd4\xab\x0d\xde\xfa\x0a\x35\x1b\x9f\xec\xed\xe5\xa2\x8e\x45\x98\xc6\xee\xdf\x74\x01\xdc\x0f\xe9\x50\xc5\xa9\xdd\x3c\x37\x82\x29\x12\x76\xb7\x0b\x22\xb8\x5c\x6e\x53\xd4\x63\xdb\x14\xec\x85\x76\x5e\x8e\xfd\xae\x75\x18\x22\x81\x81\x8b\xc9\xf3\x24\x3e\xb9\xe8\x2a\x1d\x74\x48\xd7\xfa\x6a\xc0\xc1\x24\x2c\xe0\x31\x19\xc7\xcb\xfe\x14\x2d\x18\x3d\x51\x0c\xda\x95\xce\x9b\xf6\x8b\x79\xdc\x4c\xe8\x13\xca\x7d\xb8\xce\xee\x39\x03\xd2\x96\xc3\x50\xb8\x3a\x84\x11\x19\x9e\xe2\x39\x01\xdf\xba\x2d\x9b\x0e\x32\x28\xb2\xdc\x2e\xef\x62\xf6\xb7\x2c\xf1\x2d\xb2\xdb\x4a\x22\x69\x66\x11\x92\x46\x46\x0e\x6a\x8d\xe5\xcc\xc9\x76\xd6\x7e\x7d\xc6\x20\x8d\x72\xf1\xc7\xb9\x9d\x88\xea\x78\xe2\x84\x0d\x84\x17\xd7\xd7\x02\xda\xd5\xdd\xe1\x79\x64\x75\x69\x76\xed\x62\xaa\x33\x90\x0f\x4c\xe6\xab\xcf\xe7\x59\xc9\xac\x7f\xa4\x33\xc8\x91\xc4\x55\xc8\x19\xf6\xa7\xa0\xb5\x2c\xdf\xbd\xc7\xda\x99\xa6\x92\xea\x2b\x70\xa0\xc4\xf0\x2e\x54\x71\xc9\x6e\xfd\x69\x22\x8c\x4f\xad\x59\xca\x6e\xa4\x90\x35\xbe\xed\xca\x70\x1b\x04\x76\xf4\x3e\x94\x86\x41\x19\xb9\xff\0\x84\x7e\x5b\xe6\xbc\x99\xa4\x27\0\x6e\xe1\x46\xfc\x55\x94\xb3\xfb\x66\xb7\x7e\xad\x71\x2c\x41\x56\x33\x88\xdf\x6e\x49\x5e\xa7\xf2\xa6\x21\x9a\x75\xcd\xc4\x97\x3a\x77\x9b\x33\x38\x63\x2a\x13\x9e\x1b\x1d\x0f\xbd\x45\xad\x66\x74\xd5\xc3\xbb\x15\x89\xa3\x2a\x37\x70\x3a\x03\x47\x50\xe9\xfd\x76\x1f\x3a\xcd\x3d\xf9\xb4\x8e\x09\x67\xb7\xb7\x8d\x76\xa8\x9b\x67\x27\xf8\x89\xef\x4d\xb7\x37\x13\x35\xa5\x95\xe3\x91\x0b\x4b\x20\x38\x93\x24\x85\x1c\x29\x22\x97\x40\x2f\x69\x36\xd1\xdb\x6b\xd7\xb1\xc0\x4e\xc1\x1a\x60\x16\xce\x3a\xf1\x4d\xf1\x04\xcf\x61\x76\x97\x6b\x90\x25\x85\xe2\x3f\xef\x63\x2b\x4b\x76\x3d\x91\x47\x4f\x59\x05\xc5\xbe\x93\x29\x2c\xd0\x4e\x65\x27\xfd\x90\x32\x3f\x53\x49\x1d\xb4\x32\xd8\x69\xc6\x64\x0f\xfe\x96\xc8\x49\xf4\xcb\x71\x55\x7b\x08\xb1\x0e\x9d\x6b\x70\xfa\xb4\x92\xc7\xbc\xa4\x8c\x14\x93\xf7\x78\xed\x5a\xfa\x23\x33\xe9\x16\xac\xed\xb8\xf9\x63\x93\xf4\xa9\x6c\xa4\x64\x69\x76\x77\x73\xc9\x7a\x6d\xef\x9a\xde\x31\x74\xe0\xa8\x8c\x1c\x9f\x5c\x9a\x4d\x6a\xc6\x3b\x2b\x4b\x08\x52\x56\x45\x17\x20\xb4\xb9\xc1\x04\xf7\xa7\x7d\x45\x6d\x0b\x76\x29\x6e\x97\x8a\x46\xb0\xf7\x0d\xda\x36\x90\x10\x7f\x0a\x8b\x42\xbc\xb5\xb7\xfb\x6c\x73\x4f\x14\x4d\xf6\x97\x20\x33\x01\xc5\x21\xe8\xba\x9a\xf3\x30\x96\xd1\xda\x06\x0c\x59\x0e\xd2\x0f\x07\x8a\xe7\x27\x01\xbc\x14\x81\x78\xc6\xd1\xc7\xae\xfe\x68\x41\x23\x45\x34\x44\x69\x23\x7b\x8b\xbb\x8b\x80\x84\x30\x57\x6f\x97\x3f\x4a\xa9\x16\x9f\x15\xee\xbb\xa8\xf9\xa6\x45\xd9\xb3\x05\x18\xaf\xf0\xd1\xcc\x1c\xa4\x06\xd5\x2d\x2e\xb5\x68\xa2\x2c\x57\xec\xb9\xf9\x8e\x4f\x43\x56\x34\xed\x0e\x19\xac\x6d\xe4\x9a\x7b\x87\x56\x45\x63\x19\x7f\x97\xa7\x4c\x53\x72\x12\x46\xe1\x50\x23\xda\xa3\0\x0c\x01\x5c\x86\x95\xfd\x90\xb6\x48\xb7\x8d\x22\xcc\x09\xc8\x05\xf1\xd7\xda\x92\xbd\xb4\x1c\xad\xd4\xd7\xbc\xbb\x56\xb0\x82\xdb\x4e\xf9\xda\xe0\x79\x71\xe4\x1f\x94\x0e\x09\x39\xa8\xf4\xb8\xdf\x48\xbc\xfe\xcf\x6c\xbc\x12\x8d\xd1\x3e\x3f\x8b\xb8\x34\xfa\x07\x5b\x94\x64\x81\x9f\x49\x9e\x32\x8c\x73\x7e\x72\x31\xd4\x66\xb7\x6d\x74\xab\x2b\x59\x44\xb0\x40\x15\xd7\xa1\xc9\x38\xa1\xb0\x48\xb9\x8a\x4a\x82\xc4\x3e\xa7\xb5\x34\xb0\x14\xc4\x31\x40\x20\xe7\xd6\x91\xd7\x04\x50\x03\x4f\xde\x39\xe9\x4a\x39\x61\xc7\x02\x80\x15\xc7\x03\x35\x1e\x33\x92\x39\xfa\xd0\x02\x9e\x83\x8a\x13\x3f\x31\xe9\xcd\0\x23\x8e\x2a\x36\x1c\x9c\x83\x4c\x03\x04\x63\x8c\xd2\x37\x39\x38\xa0\x05\x61\xc8\xa6\xf6\xc6\x3a\xd0\x03\xfb\x53\0\xca\x62\x80\x11\x7a\xe4\xf5\xa4\x65\x39\x22\x81\x17\xe9\x68\x18\x66\x8a\0\x29\x7a\x76\xa4\x02\xfd\x28\xa6\x01\x45\x20\x32\x3f\xb3\x26\x38\xc9\x5e\x2e\xcc\xdd\x7b\x55\xa4\xb4\x75\xd4\x6e\x67\x24\x6d\x95\x15\x54\x7b\x81\x55\x72\x52\x33\xd7\x47\xb8\x8d\x6d\x88\x58\x66\x31\x21\x42\x92\x67\x68\x39\xce\x45\x32\x5d\x3e\x4b\x4b\x79\x5e\x79\x63\x8d\x8d\xc2\xcb\x1b\x80\x76\xee\x23\x18\x3e\x82\x9d\xee\x4d\xac\x16\x91\xc9\x7d\xaa\xcb\xe7\xc9\x1c\xaa\x6d\x8a\x33\x45\xca\xae\x4f\x03\x3e\xbd\x4d\x5a\xb6\xd3\xee\xc3\x47\x14\x86\x04\x86\x12\x08\x74\x5f\x9a\x4c\x74\x07\xd2\x87\x64\x34\xba\x97\xf5\x1b\x53\x77\x64\xf0\x2b\x6d\xdd\x8e\x4f\xb1\x07\xfa\x50\x6c\xc9\xd4\x63\xb9\x2c\x36\xa4\x46\x3c\x77\xeb\x9c\xd4\xdc\xab\x14\xe5\xd2\x24\xc0\x68\xa4\x4f\x31\x67\x69\x40\x75\xca\x9c\xf6\x34\xb1\x5a\xc9\x67\x71\x71\x79\x7b\x3a\x3c\x72\x20\x12\0\x84\x63\x1c\x0c\x7b\x73\x4e\xe4\xda\xc2\x78\x6a\xcf\xc8\xb6\x79\x98\x30\x69\x5b\xe5\xdc\x39\x08\x38\x5f\xf3\xef\x56\x75\x4b\x07\xbd\x78\x1e\x29\xbc\x97\x80\x96\x53\xb7\x3c\x9a\x4d\xea\x34\xb4\xb1\x1a\xe8\xe2\x54\x9c\xdd\xcb\xe6\xcf\x36\x3f\x78\x17\x1b\x71\xd3\x15\x25\xa6\x9d\x3a\x5c\x2c\xd7\x77\x66\x72\x80\x85\x50\xbb\x40\xcf\x72\x07\x5a\x2e\x16\x2c\xde\x5a\x2d\xd3\x42\x5d\x88\xf2\x64\x12\x0c\x77\x22\xa1\xbb\xd3\x9e\x4b\xaf\xb4\xdb\x5c\xb5\xbc\xa5\x76\xb9\x0a\x08\x61\xdb\xaf\x7a\x57\x1d\x86\xb6\x8f\x19\x86\x15\x49\xa4\x49\x61\x62\xcb\x2f\x56\xc9\xeb\x9c\xd3\x57\x43\x8b\x65\xd0\x92\x69\x5d\xae\x80\x0e\xc7\x19\xe0\xf6\xa7\x71\x72\x92\x4d\xa5\x07\x9d\xe5\x86\xe6\x68\x0c\x8a\x15\xc2\x11\x86\xf7\xe7\xa1\xa5\xfe\xc7\xb7\x0f\x6c\xd1\x34\x91\x9b\x71\xb5\x76\xb7\x0c\x3d\x0f\xad\x17\x0b\x12\x0d\x26\xdc\x69\x9f\x60\xcb\xf9\x59\xce\x73\xcf\x5c\xd4\x77\x1a\x34\x33\x5c\xbd\xc0\x9a\x78\xa4\x90\0\xc6\x37\xc6\x40\x18\xc5\x17\x0b\x0e\x97\x47\xb5\x7b\x78\x22\x4d\xf1\x79\x1f\xea\xda\x36\xc3\x0c\xf5\xe7\xde\x84\xd1\xad\x12\xde\x78\x42\xb1\x5b\x8c\x79\x84\xb1\x24\x91\xde\x95\xc3\x95\x0b\x26\x8f\x6d\x22\xc7\xf3\x4a\xac\x8b\xb7\x7a\xc8\x43\x11\xe8\x4f\x7a\x7b\xe9\x56\x6d\x69\x1d\xb7\x96\x44\x71\x9c\xa9\x53\x82\x0f\xae\x7d\x68\xb8\x59\x0f\xb2\xd3\xad\xac\xdd\xde\xdd\x08\x77\x03\x71\x2c\x49\x6c\x7d\x6a\x5b\xab\x58\x6e\xd1\x56\xe2\x30\xe1\x18\x30\xcf\x62\x3a\x52\xbe\xb7\x1d\x80\x5a\xc0\xb7\x2d\x70\xb1\x81\x33\x2e\xd2\xdd\xc8\xa8\x9f\x4d\xb3\x7b\x61\x6e\xf0\x29\x8b\x76\xed\xa7\xd7\xd6\x9d\xc2\xc4\x91\xda\xc2\x8b\x22\xa4\x6a\x04\xbf\x7c\x0f\xe2\xa7\xc5\x12\x43\x1a\xc7\x1a\x85\x45\x18\0\x76\xa4\x31\xb1\x43\x1c\x5b\xbc\xa4\x54\xde\xc5\x9b\x03\x19\x3e\xb4\x93\xc1\x0d\xc2\x04\xb8\x89\x24\x50\x72\x03\x2e\x45\0\x43\x1d\x85\x9c\x4e\x1e\x3b\x68\x51\x97\xa1\x08\x01\x14\x36\x9f\x68\xcc\x59\xad\x61\x25\x8e\x49\x28\x39\xa2\xec\x56\x44\xc8\x8a\x8a\x11\x14\x2a\x81\x80\x07\x41\x55\x6d\x74\xf4\xb6\x59\xa3\x18\x68\x64\x7d\xea\x84\x7d\xd3\xdf\xf5\xa0\x76\x2d\x7e\x34\x81\x14\x31\x60\xa0\x13\xd4\xe3\xad\0\x26\xc5\x24\xe4\x0e\x78\x3c\x75\xa5\xc0\x03\0\x70\x3b\x50\x01\x4c\xf2\xd3\xae\xd1\xf9\x50\x02\xe0\x76\x14\x87\x93\xd2\x80\x0c\x01\xd0\x50\x69\x80\x94\x9d\x69\0\x1a\x69\x19\xa0\x62\x50\x69\x88\x4c\x52\x01\x40\xc0\xe3\xa5\x21\xa0\x42\x52\x62\x80\x03\x4d\x34\0\x84\x51\xd2\x80\x1b\x46\x39\xa0\x04\xe2\x93\0\x1a\0\x4c\x0c\xe4\x50\x68\x02\xd9\xa5\x14\xc0\x29\x68\0\x14\xa2\x80\x0c\xd2\xd2\0\xe6\x8c\xd0\x05\x7b\x5b\xa4\xb8\x92\x74\x50\x41\x85\xca\x36\x7b\x91\x49\x73\x78\x2d\xcc\xbb\xa3\x90\x88\xa3\xdf\xbb\x1c\x1f\x61\x4e\xc2\xb9\x34\x13\x2c\xb1\xab\xa9\xfb\xc0\x1c\x53\x9b\xcb\x7f\x91\xf6\xb6\xef\xe1\x3d\xe8\x01\xb0\x79\x08\x4a\x42\x11\x7d\x97\x02\x9c\x65\x8c\x1c\x17\x5c\xf6\xe6\x90\x0b\x24\x9b\x17\xe5\xc1\x72\x09\x55\x27\x1b\x8d\x32\xc6\xe9\x2e\xed\x96\x54\x04\x67\xa8\x3d\x41\x1d\x45\0\x49\x34\xf1\x43\x83\x2c\x8a\x99\xe9\xb8\xe2\x92\x49\xe2\x45\x05\xe4\x55\x53\xd0\x93\x8c\xd0\x03\xd1\x95\xd0\x32\x10\x54\x8c\x82\x3a\x1a\x89\x2f\x2d\x9e\x53\x12\x4f\x1b\x49\xd3\x68\x60\x4d\0\x24\x9a\x85\xa4\x6c\x55\xee\x22\x52\x0e\x08\x2e\x06\x0d\x4c\x93\x44\xd2\x34\x6a\xea\xce\xa0\x12\x01\xe4\x67\xa5\x16\x0b\xa1\x62\x96\x39\x41\x31\xba\xb8\x04\x82\x54\xe7\x9a\x86\x6d\x46\xce\x09\xc4\x32\xce\x8b\x27\xf7\x73\xd3\xeb\xe9\x45\x82\xe4\x12\x6a\xf0\xc7\xa8\xfd\x8c\x11\xb8\xc7\xb8\x36\x7b\xfa\x7e\x5c\xd4\x7a\x66\xb9\x6f\x71\x14\x2b\x3c\xa8\xb3\xc9\xc1\x51\x9c\x67\xd2\x9d\xb4\x17\x36\xa5\xbb\x9d\x52\xce\xd2\x5f\x2e\x79\x82\xbe\x33\x8c\x13\x81\xef\x8e\x94\x5d\x6a\xd6\x76\xa1\x0c\xb2\x81\xe6\x2e\xe5\xc0\x27\x70\xf6\xc5\x2b\x30\xba\x1f\xa8\xce\x61\xb0\x96\x74\x70\x85\x57\x21\x98\x12\x07\xe1\x50\xcf\xab\xda\x5a\xb8\x8e\x57\x26\x40\xa1\x8a\xaa\x92\x70\x7b\xd0\x95\xc7\x7b\x09\x1e\xbb\x63\x2c\x88\xa9\x21\x21\xc8\x55\x6d\xa4\x2e\x7d\x33\xeb\x4b\x75\xac\xdb\x5b\x4c\xf1\xb0\x91\xfc\xbe\x64\x64\x42\xc1\x3e\xa6\x9f\x2b\x17\x32\x1f\x71\xaa\xdb\xc2\x63\0\x3c\xcd\x20\xdc\xab\x12\xee\x38\xf5\xfa\x53\x62\xd6\xad\x24\xb4\x92\xe5\x4b\x6c\x47\xf2\xf1\xb7\x92\xde\x80\x7e\x34\xac\x17\x44\x96\x3a\xac\x37\x73\x34\x21\x24\x86\x60\x37\x6c\x91\x70\x48\xf5\xa6\x6a\x77\x12\xc3\xa8\x69\xf1\xc6\xc4\x24\xac\xe1\xc0\x1d\x70\xb4\x58\x2f\xa1\x99\x26\xab\x24\x51\x69\xf2\xc7\x24\xf3\x24\x92\xba\xb0\xda\x03\x3f\x5c\x0c\x7d\x6a\xf1\xd7\x62\x48\x67\x69\xe1\x96\x29\x60\x50\xcd\x13\0\x09\x07\xb8\xa6\xd0\xae\x3d\xb5\x7c\x5a\x2c\xdf\x65\x9b\x32\x3e\xd8\xd3\x03\x2f\xc6\x73\xec\x29\x83\x5b\x55\x82\xe9\xa5\x82\x48\xe5\xb6\x50\xcf\x19\xc7\x43\xd3\x06\x8b\x0e\xe2\x43\xac\xf9\x8f\x6f\xe6\x5b\x49\x14\x77\x0e\x56\x37\x7e\x33\xc7\x19\x1e\xf5\x6a\xde\xf0\x5c\x5d\xdc\x42\xa8\x76\xc0\x40\x2f\x9e\x09\x3d\xa9\x58\x69\x99\x73\x4b\x33\xf8\x9c\x2c\x91\xb8\x86\x28\xb7\x0f\xde\x60\x01\x93\xf3\x63\xbf\xa5\x4a\x75\xa9\x04\x42\xe5\xac\xd8\x59\x96\xc7\x9b\xbc\x67\x19\xc0\x3b\x7d\x29\xd8\x57\xb1\x34\x7a\x94\xb3\x6a\x72\xda\x43\x6d\xb9\x62\x2b\xbe\x4d\xfc\0\x46\x6a\x3f\x12\x5c\x5c\xdb\x5b\x44\x6d\x47\x26\x55\x05\xb7\x63\x1c\xf4\xfc\x68\x4b\x50\x6f\x46\x54\x5b\x8b\xf5\xd6\x6e\x56\x08\x15\x9d\xa2\x42\xca\xcf\x85\x4e\x2a\x56\xd7\x37\x59\x40\xd1\xa2\xad\xc4\xcc\x50\x23\xb6\x15\x48\xea\x49\xf4\xa2\xc2\xbb\x44\xda\x5e\xa4\xf7\x37\x12\x5b\x5c\x22\x2c\xa8\x37\x03\x1b\x6e\x56\x1e\xb4\x6a\x37\xd7\x30\xdf\xdb\xda\xda\xc4\x8e\xd3\x2b\x1c\xb1\xc0\x5c\x51\x6d\x47\x7d\x0a\xad\xac\x4f\x0c\x53\x25\xc4\x48\x2e\x61\x91\x15\x80\x3f\x2b\x06\x3d\x45\x5e\xba\xbc\x78\xb5\x0b\x6b\x65\x50\x56\x65\x72\x49\xea\x30\x28\xb0\x5c\xcb\x5d\x5b\x50\x3a\x4b\xdf\xbc\x70\x2a\xe0\x6c\x1c\xe4\x9d\xd8\xff\0\x1a\xb0\x35\x0b\xdb\x6b\x98\x45\xfc\x71\x08\xae\x0e\xd5\x28\x4e\x50\xe3\x80\x69\xd9\x05\xd9\x13\x6a\x5a\x87\xd8\xce\xa2\xb1\xc5\xf6\x5c\xe4\x46\x73\xb8\xae\x71\x9c\xd3\x6e\x75\x59\x9a\xf6\x48\x61\x9e\x1b\x70\xaa\xac\x9e\x68\x3f\xbc\xc8\xcf\x5e\xd4\x59\x0a\xec\x5b\xdd\x5a\x55\x68\x21\x32\x45\x6d\x2b\xc5\xe6\x3b\xb7\xcc\xa3\xd0\x0c\x7a\xd4\x69\xac\x5c\x36\x9f\x70\xc0\x2c\xb2\x42\xe1\x4c\x88\x0e\xd2\xa7\xf8\xb1\xd7\x8a\x2c\x1c\xcc\xb5\xa4\x5d\xcb\x3c\xcc\x8d\x75\x0d\xd4\x7b\x72\x19\x46\xd6\x07\xd0\x8a\xd4\xa4\xca\x5b\x08\x78\xed\x9a\x43\x52\x31\x0f\xbd\x14\xc6\x04\x53\x68\x10\x94\x74\xa0\x62\x52\x74\xa0\x04\x22\x92\x80\x10\xf4\xa4\xed\x40\x82\x90\xd0\x02\x52\x50\x05\xbc\xe6\x94\x71\x4c\x05\xed\xc5\x25\0\x28\xa2\x80\x16\x8f\xe7\x40\x0b\x45\x20\x32\x74\x86\x54\xbb\xd4\x43\x10\x3f\xd2\x0f\xf2\xa8\x75\x26\x2d\x79\x7b\x83\x91\xf6\x4e\x9f\x9d\x57\x52\x3a\x10\xac\x30\xdb\xdb\xe9\xb3\xda\x16\x13\x49\x22\x2b\x36\xee\x5c\x11\xce\x45\x3e\xc2\x38\xa2\xb6\xbf\xbb\x9b\x7e\xf4\x79\x17\x2a\x79\x51\xe8\x3f\x1a\x62\xb1\x04\x11\x9b\x7b\xed\x3f\x6c\x2b\x01\x76\x39\xc4\x9b\x99\x86\x3b\xd4\xf6\x1a\x74\x57\x5a\x75\xdb\x05\x3e\x73\x33\x85\x6c\xf2\x30\x78\x03\xf2\x14\x37\xd4\x12\x25\xd2\xa7\x6d\x4a\xfe\x39\x9b\xa5\xac\x41\x71\xdb\x79\xeb\x8f\xc2\xac\xe8\x99\x17\x1a\x80\x07\xe5\x17\x2d\x8f\xaf\x14\x98\xd1\x13\xad\xbc\xba\xcd\xd8\xd4\x02\x32\xa2\x20\x8c\x49\xd0\x29\xea\x47\xe3\x59\xf6\xd1\x2d\xcf\xf6\x74\x52\xae\xeb\x7f\x3e\x5f\x2c\x37\x74\x03\x23\xf9\x53\x11\xd2\x4b\xb6\x0b\x49\x0a\xae\x11\x10\x9d\xab\xc7\0\x76\xae\x51\x58\x05\xb1\x97\xfd\x1a\x30\x67\x42\x16\x3f\xbe\x06\x79\xdc\x69\x44\x72\x34\x60\xb7\x8a\x4b\x6d\x69\xde\x35\x67\xf3\x64\xe4\x8e\x78\x19\x14\xc9\xe4\x7d\x3a\xda\xcb\x51\x0a\x4e\xeb\x6f\x29\xbf\xde\xdb\x95\x3f\x9d\x1b\xe8\x1b\x6a\x6d\xe8\xf6\xdf\x64\xd3\xe2\x88\xfd\xe0\x32\xc7\xdc\xf2\x6b\x1e\x29\x6d\xa1\xd3\xaf\xed\xee\x82\xfd\xa9\xdd\xc1\x46\x1c\xbb\x1f\xbb\x81\xf9\x51\xb8\x3d\x37\x1f\x08\x16\xfa\x9d\xaa\x5d\x1f\x9c\xd8\x84\x39\xee\xd9\xfe\x74\xc5\x89\x57\xc2\x91\x32\x26\x0e\xf5\x6c\x81\xce\x77\xf5\xfa\xd3\x11\x14\xa0\xdb\xdd\xdf\xa5\xcd\xe3\x5b\xf9\x8e\x58\x27\x94\x1b\xcc\x52\x38\xc6\x7a\xfa\x62\xad\xd9\x5a\x88\xaf\x34\xc5\x28\xe4\x2c\x2c\x47\x98\x39\x5e\x46\x3e\x9d\x68\x03\x4b\x5e\x52\xda\x3d\xd2\xa8\xc9\x29\xc0\xfc\x6a\xb5\x94\x4c\x35\x89\xd9\x93\x8f\xb3\xa0\x07\x1f\x5c\x8a\x95\xb1\x4f\x72\x9c\x30\x38\xf0\xf5\x92\x88\xd8\x30\xb9\x52\x46\xde\x71\xbc\xf3\x51\xde\x79\xd2\x4b\x7f\x14\xa6\xe1\x24\x62\x44\x70\xc1\x1f\x0e\x31\xc1\x27\x1c\xd3\x24\x93\x4e\x69\x2c\xe5\xb5\xba\x78\x25\x92\x27\xb5\x58\xfe\x44\x25\x95\x87\x51\x8a\xac\x90\x4d\x75\x69\x71\x2a\x45\x2a\xf9\x77\xed\x23\xa2\x1f\x9c\x0c\x73\x8f\x71\x4f\x60\xf2\x34\xf4\x88\x12\x4d\x4b\xcf\x8e\x2b\xa6\x54\x8f\x1e\x6d\xc3\x1e\xa7\xb0\x06\xae\x6a\x50\x4b\x26\xa7\xa7\x3a\x21\x64\x8d\x9c\xb1\x1f\xc3\xf2\xf1\x53\xd4\xae\x86\x42\xdb\x5d\xdb\xdb\x58\x32\xda\xc9\x21\x86\x79\x1d\xd3\x1c\xe3\x27\xfa\x54\x97\xf6\xd7\x1a\x9f\xdb\x2e\x96\xda\x48\xc1\x80\x45\x12\x38\xc3\x37\xcd\x92\x71\x4f\x4d\xc5\x67\xb1\x6f\x5b\x8e\xe8\xc1\x6a\x23\x59\x5a\x05\x3f\xbf\x48\x4e\x18\x8c\x71\x59\xd1\xd8\x5c\x1f\xed\x14\x8a\xca\x48\xa3\xb9\x84\x18\xc3\x36\x72\x47\xa9\xcf\x5f\x6a\x13\xd0\x1a\x77\x34\xf5\xa8\x4f\xf6\x2e\xf3\xb5\x65\xb7\x0b\x22\x13\xd9\x96\xa4\xd0\x2d\xde\x2d\x3d\x5e\x6f\xf5\xd3\x93\x2b\xfd\x4d\x4f\x41\xf5\x1b\x2d\x93\xc9\xac\x49\x2b\x2f\xee\x5e\xd7\xca\xdd\x9e\xf9\xac\xf3\x6d\xa8\xc9\xa7\x0d\x2e\x4b\x60\x14\x61\x4c\xfb\x86\x36\x83\xd7\x1d\x7a\x53\xba\x06\x99\xa3\xa7\xd9\xc9\x6f\xa8\xde\xc8\xcb\x88\xe5\xf2\xf6\x1c\xf5\xc2\xe2\x8d\x76\xda\x6b\x9b\x1d\x96\xc0\x34\x8a\xea\xc0\x13\x8c\xe0\xd2\xbe\xa3\xb6\x83\x2c\xad\x67\x5d\x46\xe2\xea\x64\x09\xe7\x46\x83\x01\xb3\x82\x3a\xd6\x73\x68\x93\x2c\x10\xb9\x8e\x39\xa4\x86\x57\x6f\x29\xcf\x0e\xac\x7d\x7d\x69\xdc\x56\x2f\x69\x36\x72\x45\x3c\x93\xc9\x6f\x0d\xb0\x61\xb5\x63\x8c\x02\x40\xee\x49\x15\x5f\x56\xf3\xc6\xb9\x62\xd6\xaa\xad\x20\x47\x3b\x58\xe0\x11\xc7\x7a\x3a\x83\xbd\x88\xe4\xd2\x6e\xee\x60\xbb\x96\x53\x1a\x5c\xce\xca\xca\xa3\x90\xa1\x7a\x0c\xd4\xc9\x69\x7f\x3e\xa5\x05\xd5\xd0\x8a\x34\x89\x59\x76\x2b\x64\xf2\x3a\xd3\xba\x0b\x32\xbe\xa1\x67\x25\x9f\x85\x1a\xd9\xc8\x32\x20\x19\x23\xa7\xdf\xcd\x4a\xb6\x17\xb7\x73\xc0\xda\x83\x44\x63\x83\xe6\x01\x33\x97\x6e\xc4\xd1\x75\x6b\x8a\xce\xf6\x18\xba\x5d\xff\0\xd9\x06\x9e\x65\x8b\xec\xb9\xc6\xfe\x77\xed\xce\x71\xe9\x4f\xbc\xd3\xaf\x1e\x67\x58\x4d\xb4\x96\xec\0\x54\x99\x73\xe5\xe0\x63\x8a\x2e\x87\x66\x34\x68\xf2\xda\xfd\x9a\x4b\x39\x63\x69\x62\x8f\xcb\x7f\x34\x64\x30\xce\x7f\x9d\x4b\xfd\x9d\x76\x6d\x5f\x37\x65\x6e\x19\xc3\x82\xa3\xe5\x18\xed\x8f\x4a\x2f\x70\xb3\x0d\x3f\x4e\x9a\x2b\xe6\xbb\xba\x68\xcc\x85\x36\x01\x12\xe0\x7e\x3e\xb5\xa5\x49\xea\x34\xac\x21\xa4\xa4\x31\x28\xa0\x62\x1a\x4a\0\x4a\x43\xc5\0\x19\xa4\xa0\x04\xa4\xcf\xa5\x02\x10\xd1\xd6\x80\x1a\x78\xa0\xd0\x02\x52\x1a\0\xb6\x3a\x52\xd3\0\xa2\x80\x16\x8a\0\x29\x69\0\xa2\x96\x80\x33\xe7\xd1\xac\xa7\x99\xe5\x92\x22\x5d\xce\x49\xdc\x6a\x54\xd3\xad\x11\x0a\x88\xc6\x0a\x79\x67\x93\xf7\x7d\x29\xdd\x93\xca\x86\xc1\xa5\xd9\xdb\xca\x25\x8a\x20\xac\x3a\x1c\x93\x8f\xa5\x4e\x96\xd0\xac\x72\x20\x45\xdb\x21\x25\x87\xa9\x3d\x68\xbb\x63\xb2\x20\x8f\x48\xb1\x8c\x0d\xb0\x01\x87\x0e\x0e\x4e\x41\x1d\x39\xab\x0d\x09\x86\x07\x5b\x35\x45\x7e\x4a\x86\xe9\x9f\x7a\x1b\x6c\x2d\x6d\x88\x74\xbb\x33\x69\x6c\xe0\x94\x33\x3b\x17\x72\x3a\x6e\x35\x2e\x9f\x6a\x2d\x2d\x84\x6c\xdb\xdc\x92\xce\xde\xac\x79\x34\x5c\x2c\x2d\xcd\x95\xb5\xd1\x06\xe2\x15\x90\x8e\x84\x8a\x94\x41\x10\xf2\xf1\x1a\xfe\xef\xee\xf1\xf7\x7b\x71\x4a\xec\x09\x0f\xbd\x57\x4d\x3a\xcd\x43\x81\x6d\x16\x1c\xe4\x8d\xa3\x14\x5d\x85\x89\xd6\x28\xd4\x30\x54\x50\x1f\x25\x86\x3a\xfa\xe6\xaa\x5e\x58\xfd\xaa\x78\x03\x4a\xa2\xde\x22\x1b\xc9\x0b\xd4\x8e\x9f\x85\x17\x06\x5e\x1c\x71\xe9\x48\x62\x8c\xbe\xf2\x8a\x5c\x7f\x16\x39\xa4\x31\xe4\x29\x20\x90\x32\x3a\x52\x02\x8e\xbc\x60\xaf\x4e\x28\x01\x4a\xab\x11\x90\x32\x3a\x67\xb5\x38\xd0\x22\x1b\x9b\xab\x6b\x5c\x7d\xa2\x64\x8b\x77\x4d\xcc\x06\x69\x62\xb8\x82\x48\x4c\xc9\x2a\x34\x60\x64\xb0\x23\x18\xa0\x2e\x46\x9a\x95\x93\xb8\x54\xba\x84\x92\x70\0\x71\xc9\xa9\xc4\xd1\x99\xfc\x90\xeb\xe6\xed\xdd\xb7\x3c\xe3\xd6\x8d\x42\xe4\x17\xba\x6c\x57\x72\x89\x7c\xc9\x61\x94\x0d\xbb\xe2\x6d\xa7\x1e\x95\x2d\x95\xa4\x56\x50\x08\xa1\x18\x5c\x92\x49\x39\x24\xfa\x9a\x77\x0b\x16\x3a\x52\xd2\x18\x62\x83\x40\x08\x7d\xa9\x28\x02\x0b\xab\x38\x2e\xc2\x8b\x88\xc4\x81\x0e\x40\x35\x36\0\xe8\x28\x01\x30\x31\x46\x29\0\x94\x86\x98\x08\x78\xa0\xd0\x02\x62\x98\xf1\x46\xd2\x2c\xa5\x14\xc8\x80\x85\x6c\x72\x28\x01\xd8\xa4\xfc\x28\x01\xae\x8a\xca\x43\0\x47\x70\x69\x31\xed\x40\x07\x7a\x28\x01\x09\0\x7a\xd3\x77\xf1\xc0\xa0\0\x9f\x5a\x6e\xf1\xe9\x40\x0d\x77\xc7\x63\x41\x61\xc7\x7c\xd3\x01\x03\x64\x1e\xb4\x06\xc9\xc6\x28\0\x63\x81\x93\x4c\xdc\x4f\x24\x71\x40\x0e\x38\x3c\x8a\x69\x23\x38\xcd\0\x14\x84\x8c\x50\x02\x13\xda\x90\x91\xeb\x40\x09\x9c\x0e\xb4\x13\x81\x40\x0c\x2d\xd0\x0e\xa6\x93\x24\x1f\x9b\x03\xd2\x98\x0a\x78\xa4\x2c\x3b\x52\x02\xdf\x7a\x5e\xf4\xc0\x28\xcd\0\x28\xa5\x34\0\x66\x97\x34\0\x52\xfb\xd2\x03\x36\x6d\x59\x51\xe5\xd9\x04\xb2\x45\x09\xc3\xc8\xa3\x20\x1e\xf5\x59\xae\xe5\xb8\xd7\x61\x8d\x0c\xbe\x40\x88\x38\xda\x46\xd6\xcf\x73\xea\x2a\xac\x43\x64\xfa\xd5\xe5\xc5\xb3\xda\x8b\x74\x2c\x1e\x40\x1b\x18\xe7\xdb\x9a\x59\x75\x57\x56\x90\x43\x6a\xf2\xac\x3f\xeb\x18\x10\x30\x7b\x81\xeb\x45\x87\x7d\x44\x9b\x57\xc8\x89\x6c\xe1\x33\xc9\x2c\x7e\x60\x19\x0b\x85\xce\x3b\xd4\x37\x3a\x9d\xdb\x49\x62\x61\xb7\x71\xe6\x31\x0c\x8c\x40\x24\x80\x78\xfc\x3a\xd1\x61\x5f\xb0\xe1\xa9\x1b\x54\x98\xac\x52\xca\xcd\x76\xd1\x05\x2f\xdf\xdb\xdb\xda\xa6\x1a\xc3\x41\xe7\x8b\xdb\x73\x0b\xc4\x81\xf0\x18\x36\xe0\x4e\x3a\xfd\x68\xb0\x5c\x23\xd4\xee\x85\xcd\xb4\x57\x16\x82\x21\x70\x4e\x0e\xfc\xe3\x03\xf9\xd6\xbd\x4b\xd0\x69\x94\xe1\xbd\x32\x6a\x37\x36\xdb\0\x16\xe1\x4e\xec\xf5\xc8\xcd\x51\x5d\x66\xe2\x68\xed\x3c\x9b\x75\x67\xba\x2e\0\x2f\xc0\xda\x7e\x94\xec\x17\x1f\xfd\xb1\x34\x50\xcc\x93\xc2\xa2\xea\x39\x16\x25\x55\x6f\x95\x89\xe8\x7e\x95\x01\xb8\xb9\x87\x5a\x12\x5f\xaa\x2f\x95\x6a\xcd\xfb\xb3\x90\xc3\x23\xd6\x9d\x85\x71\x61\xd7\x65\xf3\x21\x79\x3e\xce\xd1\x4c\xe1\x76\x46\xf9\x74\xcf\x42\x7d\x6a\xee\x9b\x7d\x77\x79\x77\x38\xd9\x1a\xdb\xc3\x2b\x21\x39\x3b\x8e\x3a\x50\xd5\x81\x32\x3d\x61\xee\xc6\xb1\xa7\xa4\x2c\x8a\x8e\x5f\0\xe7\x9e\x39\xcf\xe1\xd2\xa8\xd9\xcf\x7d\x6d\xa7\xdd\x5c\xc1\xe4\xf9\x51\x4c\xe4\x87\xc9\x2f\xcf\x3c\xf6\xa3\x4b\x06\xb7\x2d\x5d\x6b\x4e\x6e\x8c\x31\xcd\x0d\xb8\x44\x56\x2d\x28\x24\x92\x46\x70\x31\x57\xf4\x4b\xe6\xd4\x2c\x56\x69\x14\x07\x04\xab\x01\xd3\x23\xd2\x93\x56\x1a\x77\x65\x1d\x36\x38\xae\xa4\xbf\xbd\xba\x8b\xed\x12\x24\xac\x8a\xa5\x43\x10\xaa\x38\0\x7e\x35\x0d\xb4\xb6\xb8\xd4\xa2\xb7\xb7\x9e\xd9\x9a\x02\xed\x14\x8a\x02\x8e\x0f\x20\x76\xa6\x22\xaa\x46\x2e\x34\x68\xa0\x87\x48\x73\x33\xc6\xa0\x4c\x50\x01\x9c\x7d\xec\xd6\xad\xba\x34\x7e\x23\x89\x24\x62\xec\xb6\x20\x16\xf5\x3b\xba\xd0\xc1\x1b\x62\x96\xa0\xb0\xa5\xa0\x02\x90\xd2\x01\x28\xa6\x01\xcd\x27\x43\x48\x04\xa2\x80\x12\x90\xd3\x01\x28\xa4\x02\x52\x76\xa0\0\xd2\x53\x02\x37\xe5\x80\xed\x4d\0\x73\xc9\xe2\x80\x01\xcb\x0f\xa5\x21\x38\x18\x3e\xb8\xa6\0\x30\x1b\x81\xd6\x93\xee\x9c\xe7\x8a\0\x53\xd3\x8a\x63\x63\x60\xa0\x06\x83\xb9\x98\x13\xd3\xb5\x26\x70\x57\x9e\x3a\x50\x02\x12\x41\x62\x29\x71\x82\x0e\x68\x01\x32\xc0\x1c\x8e\x29\x4e\x36\xf1\xd2\x80\x11\x07\xc8\x33\x51\x9e\x7a\x0f\xa9\xa0\x07\x0f\xbc\x69\x98\x19\x19\xe9\x40\x08\xdf\xc5\x8a\x52\xbf\x30\xa6\x03\x70\x30\x40\xed\x48\x7e\x60\x4f\xb5\x02\x10\x70\x46\x7b\x8c\x50\xd8\x3c\x77\xa0\x05\x6f\xba\x69\x84\x70\xb4\x01\x78\x51\x40\xc5\xed\x45\0\x2e\x69\x7a\xd0\x01\x8e\x7d\x69\x68\x01\x47\xbd\x27\x73\x40\x18\x48\x97\x76\x90\xdc\xd9\xad\xab\xc9\xe6\x96\xd9\x20\xfb\xa4\x37\xaf\xa5\x58\xb3\xb0\x92\xda\xfe\x1c\x02\x63\x8e\xd4\x46\x5b\xd4\xe6\x9b\x21\x26\x4d\xad\xc3\x2c\xb1\x5b\xbc\x11\xf9\x8d\x0c\xca\xe5\x41\xc6\x40\xcd\x66\xcb\xa7\x34\x73\xdc\x96\xb0\x6b\x86\x99\x8b\xc6\xe1\xf0\x06\x7b\x1e\x7b\x50\x98\x34\x4d\x73\x60\xe9\x1d\xba\x35\x82\xce\x89\x1e\x3f\x76\xc5\x59\x1b\xdb\x9e\x94\x89\x63\x7f\x1c\x16\x72\x32\x19\xa4\x86\x56\x72\x8c\xfc\x80\x7a\x0c\xf7\xa7\x70\xb3\x25\x1a\x75\xc9\x60\x76\x0f\xf8\xfe\x33\x75\x1f\x77\xd6\x97\x59\xb1\x79\xa5\xba\x91\xd9\x52\x16\xb7\x0b\xbd\x8f\x01\x83\x67\xfc\xfd\x69\x5c\x2d\xa1\x01\x96\xea\x6d\x4f\x4e\x59\xd2\x35\xda\xcc\x40\x46\xdd\x9f\x97\xef\x7b\x0a\xd9\xb0\xb8\x7b\x81\x31\x75\x55\x09\x29\x45\x20\xe7\x70\x1d\xe8\x63\x45\x39\xed\xaf\x21\xd4\x6e\x27\xb5\x89\x24\x5b\x95\x50\x4b\x36\x36\x10\x31\x9f\x7a\x6e\x9f\xa4\xcf\x6f\xfd\x9f\xbc\xa9\xfb\x3f\x99\xbf\x07\xae\xee\x94\x5d\x58\x56\x63\xee\x74\x99\xe5\xb9\xb8\x99\x1d\x43\x19\x63\x96\x2c\xf4\xca\x8c\x60\xd2\x0d\x3a\xee\xf2\xed\xa6\xbd\xf2\x91\x1a\x06\x8b\x62\x12\x48\xcf\x7a\x2e\x16\x63\x6d\xb4\xab\xa5\x78\x22\x78\xed\x63\x8e\x22\x0b\x4a\x8a\x0b\x38\x1f\x87\x1e\xf5\xa3\xa6\x59\xbd\x99\xba\x2e\x54\xf9\xb3\xb4\x83\x69\xec\x71\x49\xb1\xa4\x33\x52\xb3\x9a\x7b\xab\x4b\x8b\x66\x40\xf6\xec\xc4\x87\xce\x08\x23\x07\xa5\x46\x9a\x5b\x8d\x2e\xe6\xd0\xba\xee\x99\x9c\x82\x07\x03\x26\x8b\xe8\x16\x1b\x26\x99\x73\x14\xa6\x7b\x27\x87\x7c\x88\xab\x22\xca\x09\x19\x03\0\x8c\x55\xed\x3e\xdd\xed\xad\xd6\x39\x65\x33\x3f\x25\x98\x8c\x64\x9a\x4c\x69\x58\xa7\x2e\x9b\x73\x15\xcc\xb3\x69\xd7\x09\x10\x98\xee\x78\xdd\x72\x33\xea\x29\xb0\x68\xd2\xac\xd7\x12\xcd\x73\xe6\xc9\x71\x09\x8d\xc9\x5c\x73\xea\x07\xa5\x3b\x8a\xc6\x8d\x8d\xbf\xd9\x6d\x22\xb7\xdd\xbb\xca\x40\xb9\xc6\x33\x81\x8a\xa9\xa8\x69\x93\x4f\x7c\xb7\x56\xd7\x66\xdd\xc4\x7e\x59\xc2\x06\xc8\xce\x7b\xfe\x14\x93\x1d\xb4\x25\x82\xce\xf1\x23\x51\x2d\xf9\x91\x84\x81\x8b\x79\x60\x65\x7f\xbb\xff\0\xd7\xab\xd8\xe2\x90\x0a\x29\x7b\x71\x40\xc2\x93\x9c\xd0\x01\xfa\xd2\x50\x02\x50\x7a\xd0\x02\x13\x49\x9a\0\x3a\x72\x69\xb4\0\x52\x1a\0\x4f\x4a\x28\x18\x71\x49\x40\x0d\x2a\x38\xf6\xa4\0\x73\xef\x40\x84\xd8\x31\xf4\xa4\x2a\x31\x83\x4c\x04\xda\0\xa4\xd8\xa0\x74\xe2\x80\x17\xa7\x4a\x66\xc1\x9c\xe2\x80\x02\xa3\x39\xa4\x2a\x08\xe9\xc5\0\x26\xc0\x06\x31\x40\x50\x3b\x73\x40\x07\xb1\xa4\xda\x07\x6a\x06\x14\xd2\x80\x9e\x94\x08\x0a\x83\xd6\x90\xa8\xc6\x28\x01\xa1\x71\x9f\x4a\x31\x40\x06\x05\x34\x2e\x33\x9e\x94\0\x11\xc5\x34\0\x0f\x4a\0\x5c\x7a\xd2\x10\x28\x02\xcd\x28\xfa\x73\x4c\x42\xd1\x40\x0b\x4b\x40\xc0\x52\xd0\x01\x9e\x28\x14\0\x71\x4b\x8a\x42\x17\xb5\0\xd0\x31\x71\x46\x28\x01\x47\x14\x32\xab\xa9\x47\x50\xca\xdc\x10\x46\x41\xa0\x08\x2d\xb4\xfb\x4b\x57\xdf\x04\x08\x8d\xea\x07\x35\x64\0\x28\xbd\xc4\xb4\x17\x14\xa3\xad\x03\x17\xb5\x14\x84\x38\x52\xf4\xa0\x02\x97\x8a\x06\x27\x53\x4a\x28\0\xa5\x1d\x28\x10\x62\x96\x81\x87\xb5\x14\0\xbc\xd1\x9a\0\x4a\x29\0\x94\x66\x98\x08\x68\xed\x48\x04\xa2\x98\x05\x21\xa0\x04\xa6\xd0\x01\x41\xa0\x62\x75\xa3\xeb\x40\x09\x48\x28\x10\x52\x74\xa0\x04\xc5\x27\xf2\xa0\x04\x22\x83\x40\xc4\x34\x98\xa0\x04\x22\x8a\x60\x27\xd2\x8a\0\x4f\x6a\x4a\0\x4a\x4a\0\x4a\x4a\0\x0d\x27\x4a\x04\x21\xf5\xa4\x34\0\x94\x94\x01\x62\x9c\x29\x88\x5a\x3b\x50\x30\xa5\xc5\0\x14\xb4\x80\x5f\xa5\x14\0\x76\xa5\xa0\x02\x94\x50\x02\xd1\xd6\x80\x14\x74\xcd\x2d\x20\x0e\xd4\xa2\x98\x85\xa0\x52\x18\xea\x5e\xd4\0\x01\x4b\x9a\0\x5a\x28\x10\xb4\x94\x0c\x5a\x51\x40\x07\x7a\x28\x01\x68\xa0\x02\x94\x50\x02\x52\x50\x01\xc5\x1d\x3b\x52\x01\x33\x48\x7a\xd0\x01\x49\x9a\0\x29\x09\xcf\x4a\0\x4e\xfe\xd4\x94\xc0\x3b\xd1\x40\x09\x49\x40\x07\x19\xa3\x82\x68\x01\x31\x8a\x4e\x82\x80\x0a\x43\x40\xc4\x34\xd3\xd6\x80\x03\xc5\x14\0\x94\x77\xa0\x04\x3d\x69\x0d\0\x36\x94\xd0\x03\x7e\xb4\x50\x02\x1a\x6d\x30\x0c\xf1\x49\x40\x09\x9a\x43\x40\x84\xa4\xa0\x0b\x02\x97\xa5\x30\x17\x34\xb4\x08\x29\x68\0\xed\x4b\x9a\x06\x02\x96\x80\x0a\x29\0\xa2\x96\x80\0\x33\x4b\x9c\xd0\x21\x68\xa0\x61\x4e\x14\0\xb4\x0e\x94\x80\x70\xe6\x94\x50\x20\xe6\x8f\xad\x03\x14\x11\x46\x79\x1c\xd0\x01\xd6\x94\x50\x01\xf8\xd2\xd0\x02\xd1\x9a\0\x4c\x8f\x5a\x70\xf6\xa0\0\x9a\x6e\xf5\x1c\x64\x0a\0\x5c\x8c\x67\x34\xcf\x31\x18\xe1\x58\x13\xf5\xa0\x07\x9a\x4a\0\x09\xa6\x86\x07\x90\x41\xa4\x02\x06\x0c\x48\x52\x0e\x0f\x3e\xd4\x53\x01\xaf\x22\x20\xcb\x30\x50\x4e\x39\x3d\xe9\x49\xe2\x80\x12\x82\xd8\xff\0\x1a\0\xab\x16\xa3\x69\x2c\xa2\x38\xee\x23\x67\x3c\x60\x30\x24\xd4\x9f\x69\x87\xcf\xf2\x3c\xc5\xf3\x71\x9d\x9b\xb9\xc7\xd2\x9d\x98\x5d\x04\xd7\x10\xc0\x54\x4d\x2a\x21\x73\x85\xdc\x71\x93\xed\x51\x5d\xdf\xda\xda\x60\x5c\x4c\x88\x4f\x40\x4f\x34\x59\xb0\xba\x16\x3b\xdb\x79\x20\x69\xd2\x54\x31\x28\x24\xb0\x3c\x0a\x86\x3d\x5e\xc1\xd8\x2a\xdd\x46\x49\xe9\xf3\x75\xa2\xcc\x57\x45\xce\x31\x9a\xa1\x26\xb3\xa7\xa3\xb2\x35\xca\x06\x53\x82\x3d\x0d\x09\x5c\x6d\xa4\x5b\x8e\x54\x95\x03\xc4\xea\xea\x7a\x10\x72\x0d\x45\x6f\x79\x05\xc4\x92\x47\x0c\x81\x9e\x23\x87\x03\xb5\x01\x72\x6a\x4e\xdc\xd2\x18\x51\x9a\0\x4a\x4a\0\x29\x0d\0\x25\x06\x80\x12\x92\x98\x08\x69\xb8\xcd\0\x14\x9f\x5a\0\x4a\x43\x40\x84\xa4\x34\x01\x60\x1a\x51\xc7\x41\x4c\x05\xe9\x47\xd6\x81\x06\x69\x45\x03\x01\x4b\x40\x85\xa3\xa5\x20\x32\x2e\xb5\x76\xb5\x9e\xf4\x48\xaa\x56\0\x9b\x07\x23\x71\x20\x9e\x4d\x68\x59\x5d\x25\xdd\xaa\x4f\x19\xf9\x58\x77\x15\x4d\x75\x12\x7a\xd8\xab\x0e\xab\x14\x76\x71\xcb\x71\x32\xb3\x39\x20\x6c\x53\xf3\x60\xf6\x1d\x6a\xed\xa5\xd4\x57\x71\x09\x21\x6d\xcb\x9c\x7d\x29\x34\x09\x94\xe0\xd5\xe3\x93\x50\xb9\xb7\x6e\x12\x11\x90\xd8\x3c\xf1\xcd\x49\x6f\xac\x5a\x5c\x4e\xb1\x46\xe4\x33\xfd\xdd\xca\x46\xef\xa6\x69\xf2\x87\x31\x0e\x9d\xaa\x7f\xa2\xa3\x5d\x36\x65\x79\x59\x14\x28\xe4\xe0\xfa\x55\x8b\x9d\x5e\xd6\xde\x56\x8d\xcb\xb6\xcf\xbe\x55\x49\x09\xf5\x22\x8b\x05\xc2\x7d\x5a\xda\x09\x63\x8c\x07\x91\xe4\x4d\xe8\x23\x52\xdb\x87\xb5\x59\xb0\xbb\x8a\xf6\x1f\x36\x12\x48\xc9\x04\x11\x82\x0f\xa1\x14\xac\x3b\xf4\x20\x9b\x56\xb7\x8a\x09\xe6\x7d\xc1\x6d\xe4\xf2\xdc\x63\x9c\xd4\x7a\xae\xa4\x89\x6f\x24\x68\xee\x8f\xe5\x09\x44\x88\xbb\xb6\x8d\xc0\x50\x90\x9b\x2b\x4b\xab\xdc\x7d\xaa\xf6\x01\x13\xa8\x8e\x12\xc8\xdb\x3e\xe9\xda\x4e\x49\xf4\xe3\x8a\x9f\x4e\xd6\x84\x91\xda\xa4\xd1\x4c\x0c\xaa\x14\x4a\xc9\x85\x66\xc7\xad\x3b\x0b\x98\xd2\xbd\xba\x8e\xce\xd9\xe7\x93\x3b\x50\x76\x1c\x9a\xad\x65\xaa\x0b\x9b\x9f\xb3\xc9\x04\xb0\x4a\x57\x78\x12\x0e\xa3\xda\x95\x8a\xb9\x07\x89\x26\x68\x85\x9a\xac\xed\x02\xc9\x30\x56\x75\x6c\x10\x2a\xac\x53\xb5\xbe\xab\x6d\x15\xae\xa0\xd7\x71\x4b\xbb\xcc\x57\x60\xdb\x40\x19\xce\x7b\x53\x5b\x12\xf7\x2d\x1d\x79\x54\x79\xc6\xde\x41\x67\xbb\x6f\x9f\xdb\xeb\x8e\xb8\xf7\xa7\xcf\xab\xca\x2e\x6e\x21\xb7\xb3\x79\xcd\xb8\x05\x88\x60\x06\x08\xcd\x2b\x0f\x98\x82\x6d\x5a\xe2\x5b\xbd\x39\xac\xe2\x2d\x04\xe1\x89\x19\x03\x77\x1c\x8f\x6c\x54\xb7\x5a\xd4\xb0\xf9\xb2\x0b\x27\x36\xf0\xb6\xd7\x90\xb0\x07\xf0\x1d\xe9\xf2\x8a\xec\x9a\xef\x59\x8e\xd2\x67\x4b\x95\xd8\xbe\x5e\xf8\xdb\x3f\xeb\x3d\xbe\xbe\xd5\x76\xd2\x49\x26\xb6\x49\x25\x8f\xca\x77\x19\xd9\x9c\xed\xa9\x68\xab\x99\x11\x41\xa8\x40\x2f\xe6\xbd\x95\xe4\x80\xc6\xcc\xaa\x08\x1d\x06\x3f\x03\x80\x0f\x1e\xb4\xab\xaa\xcc\x16\xd6\xde\xce\x15\x67\x30\x2c\x84\x4b\x26\x38\xc7\x40\x7b\x9a\x7a\x32\x75\x43\x2e\xaf\xef\xa4\xbe\xd3\x4c\x50\x34\x6b\x26\xed\xd1\xb3\x63\x24\x0e\x41\xfa\x75\xa6\x69\x9a\x65\xb5\xf4\xd7\xed\x74\x1d\xdd\x2e\x59\x43\x07\x23\x03\xf3\xa7\xb2\xd0\x37\x7a\x8c\x63\x77\xfd\x9f\xa9\xd8\x40\xf2\x4a\xf6\xee\x02\x12\x72\xc5\x0f\x38\xcf\x73\x8c\xd4\x50\x7f\x66\x89\xad\x84\x0b\x2e\x9f\x72\xae\x3e\xfa\xb7\xcc\x3b\x83\xce\x39\xa3\xd0\x5e\xa7\x53\x9e\x2b\x36\xf3\x52\x7b\x5b\xe9\x21\x91\x54\x46\x2d\xcc\xaa\x7d\x48\xed\x50\x95\xcb\x6e\xc5\x4b\x5d\x6a\x6b\x9b\x6b\x5c\x46\xa2\xe2\x59\x8c\x6e\x98\x3f\x28\x1c\x93\xf9\x62\xb3\xfc\xfb\xa6\x8a\xc9\xac\xd6\x38\x50\xdc\xb0\x0a\x4b\x60\xb6\x4f\x5f\x6a\xbb\x24\x4d\xee\x4f\x6c\xfa\x8a\x4b\xa9\x3d\x9f\x91\xb5\x26\x2c\xde\x66\x79\x38\xe8\x31\x5b\x76\x77\x0d\x77\x63\x1c\xe0\x6d\x69\x13\x38\xeb\x83\x8a\x4e\xc3\x57\x39\x86\x37\x7f\xd9\x92\xb4\x8e\x92\x95\xbd\xc2\x81\x9f\xbd\xbb\xf9\x56\x9f\xdb\xee\xec\x2e\x5e\x2b\xf6\x49\x54\xc2\xd2\x83\x18\xc6\x31\xd4\x53\xb2\x62\x57\x45\x38\xf5\xd9\x95\xa0\x95\xe7\x82\x45\x95\x86\xe8\x50\x1c\xa0\x3e\xf5\xb7\xab\xc3\x25\xc6\x9d\x34\x50\xbe\xc7\x75\xc0\x24\xe2\x93\x56\x63\x4e\xe8\xc2\x95\xa2\x8e\xd6\x28\x75\x0d\x3a\x4b\x51\x1e\x3f\x7f\x10\x07\x69\x04\x73\x91\x49\xa9\x96\x87\x5f\xfb\x6c\x64\x91\x04\x68\xed\x8e\xea\x4e\x0f\xe8\x6a\x91\x2c\x6e\xba\xed\x77\xa8\xa4\x88\x7f\x77\x6a\xc8\xb9\xf5\x66\x3f\xe0\x2a\xfe\x8b\x1c\x77\x37\x77\xd7\x37\x0a\x1e\x65\x98\xc6\x37\x73\xb5\x47\x4a\x36\x43\xfb\x41\xa8\xdb\xda\x5b\xda\xea\x5f\x67\x60\x24\x92\x12\xcf\x18\x23\x03\x8e\xb8\xaa\x05\x26\xbb\xd2\xa2\xb7\x83\x4a\x2a\xce\x8a\x3c\xd6\0\x0e\x83\xe6\xa1\x3e\xe0\xd7\x63\x67\xed\xf6\xf6\xf2\x47\x64\xec\xc6\x50\x15\x38\x43\x8c\xfd\x6a\xa6\x83\x0c\x52\xc5\x7d\xe6\x46\x8d\x9b\xa9\x07\xcc\xa0\xf1\x9a\x9d\x90\xfa\x90\x59\xdc\x2e\x9f\x0e\xab\xb3\x1e\x4c\x32\x66\x31\xdb\x71\x1c\x8f\xcf\x02\xa9\x68\xd3\xc7\x6b\x7d\x68\xc3\x76\xf9\xd4\xa4\xc5\x94\x80\x58\x9c\x83\xfd\x2a\xba\x32\x6e\x75\x74\x98\xac\xcd\x42\x8c\x62\x80\x12\x90\xf1\x4c\0\xd2\x52\x01\x0d\x25\x30\x10\xd1\xf5\xa0\x04\xa4\xeb\x40\x01\xa6\x63\xde\x80\x0a\x43\x9a\x04\x21\xa4\xa0\x0b\x19\x34\x67\x8a\x62\x14\x1a\x3e\xb4\0\xb9\xa5\xcf\x3e\xd4\0\x52\x8a\x06\x1f\x5a\x5a\0\xc0\xbc\x89\xde\xe7\x58\xc0\xce\x62\x4c\x60\x7b\x56\xbe\x9d\x22\x4d\x63\x13\xc7\xf7\x4a\xe3\x38\xc5\x37\xb1\x0b\x73\x02\x15\x48\x6c\xac\xde\x46\x96\xdd\xc6\xf0\xb2\x85\xdc\x07\xcc\x78\x22\xb5\x74\x06\x91\xe0\x95\xdd\x57\x0d\x21\xda\xea\xbb\x7c\xc1\xfd\xec\x53\x62\x5b\x95\x6f\x1d\xe1\xbe\xd4\xf6\x44\x64\x79\x63\x42\x80\xae\x41\0\x60\xff\0\x3a\xab\x03\x66\xf7\x4f\x90\x4b\x34\xc8\xad\xb4\x93\x16\xd5\x52\x46\0\x1c\x50\x03\xac\x2d\x66\xb3\x78\xaf\xf6\x48\xfb\xa4\x64\x91\x31\xca\xa9\x3c\x10\x3e\xb4\x3d\xbb\x41\x25\xdc\x73\x35\xde\xf9\x1d\x8a\xa4\x43\xe5\x90\x1f\xc0\xd3\x15\x8d\x0b\x0b\x46\x87\x54\xb7\xfd\xdb\x04\x8e\xcf\x66\x5b\x9c\x1d\xdd\x33\x56\x74\x58\xde\x2f\xb6\x6f\x42\x9b\xae\x5d\x86\x46\x32\x38\xe6\xa5\x96\x91\x4e\xff\0\x4e\x96\x7d\x6b\xcb\x28\x4d\xa4\xd8\x92\x43\x8e\x01\0\x8c\x7f\x2a\xaf\x6f\x63\x74\xda\x2d\xeb\x4b\x0b\x99\xd9\x56\x28\xd7\x07\x3b\x57\x1d\xbf\x3a\x77\x26\xcc\xb7\x73\x0d\xc2\xde\xdd\x81\x04\x8c\xb7\x16\xbb\x55\x82\xe4\x02\x01\xe0\xd4\xd2\xda\xca\xfa\x6e\x96\x82\x32\x5a\x29\x22\x2e\x3f\xba\0\xe7\x34\x87\x66\x5b\xd6\xd2\xe1\xb4\xe6\x16\xa1\x99\xf2\x37\x04\xfb\xc5\x7b\xe3\xde\xb2\xb4\xeb\x79\x63\xd5\xed\xe6\x5b\x5b\x94\x89\xa3\x64\x2f\x33\x64\xe7\xae\x4f\xa7\x4a\x16\xc0\xf7\x34\x75\xcb\x47\xbb\xfb\x1a\xa2\x09\x15\x27\x0c\xe0\x8c\x8d\xbd\xf3\x56\x57\x4f\xb5\x44\x75\x86\x18\xe2\x2e\xa5\x4b\x22\x80\x70\x6a\x6e\x3b\x1c\xec\x3a\x4b\x84\x5b\x56\xd3\xdd\xa6\x07\x06\x56\x90\xf9\x78\xcf\x5e\x0f\xa7\x6a\xda\xb3\xb3\x96\x2b\xcd\x41\xf6\x61\x65\xdb\xb3\xdf\x0b\x54\xd8\x92\x29\x5b\xd8\x5d\xdb\xc5\xa5\xb7\x91\xb9\xad\xcb\x89\x14\x30\xca\xee\xef\x55\x2e\x74\xab\xc9\xe2\x9d\x24\xb4\x32\xdc\xb1\x3f\xbf\x69\x78\x23\x3c\x60\x67\xf0\xa2\xe8\x56\x76\x2f\x5d\xe9\xd7\x9a\x9b\x29\x94\x7d\x9d\x60\x8c\x18\x86\x41\x26\x4e\xb9\x3d\x78\xe3\x15\xb1\x64\x66\x6b\x64\x37\x09\xb2\x5c\x7c\xca\x0e\x46\x69\x32\x92\xd4\x5b\xe8\x9a\x7b\x19\xe1\x8f\xef\x3c\x6c\xab\xf5\x22\xb1\x6e\x74\xdb\x93\x14\x11\xcb\x67\x15\xe4\x6b\x0a\xa6\x37\x04\x68\xd8\x0f\x5e\xe2\x84\x0d\x0b\x1e\x97\x7d\x04\x5a\x73\x29\x49\x65\xb6\x66\xdc\x19\xbb\x37\x18\xcf\xb0\xa9\x22\xb5\xd5\x2d\x25\xbb\xfb\x32\x40\x56\x79\x9a\x45\x66\x63\x91\x9f\x6a\x77\x4c\x56\x68\x74\x7a\x45\xc2\xe9\xf3\x01\x72\x56\xf6\x67\xf3\x5a\x45\x38\x19\xf4\xfa\x54\x77\x16\x5a\x96\xa2\x91\xdb\xde\xc7\x6f\x1c\x4a\xc1\x9d\xd4\xee\x27\x1e\x9e\x94\xae\x82\xcc\xd0\xb5\x9a\x77\xbc\xb9\x8e\x40\xbe\x54\x7b\x42\x10\x0e\x7a\x72\x0d\x41\xad\xe9\x67\x51\x10\x6c\x70\x8d\x1b\x7c\xc4\xf7\x53\xd4\x52\xd9\x95\xba\x19\x6f\xa4\x79\x3a\xc4\xb7\x7b\x97\xca\x23\xe4\x41\xd9\x88\0\x9f\xd2\xa0\x6d\x1e\xe5\x2c\x6d\xd2\x19\x22\xf3\xa1\x9c\xca\x37\x67\x69\xe4\xff\0\x8d\x3b\x8b\x94\xb5\x6b\x60\xf1\x0b\xd0\xec\xa4\xdc\x39\x60\x47\x60\x46\x2a\x7d\x3a\xdd\xad\x6c\x61\xb7\x72\x19\xa3\x50\x09\x14\x98\xec\x66\x0d\x1a\xe0\xac\x91\x34\xf1\xf9\x6d\x71\xe7\x29\xc7\xcc\x0e\x79\xab\x77\xba\x68\xbb\xbc\x49\x1d\xbf\x76\x22\x68\xd9\x7b\x9c\xd3\xb8\xac\x55\xb7\xd3\x2f\x55\xe2\x8a\x4b\x98\xc5\xbc\x24\x10\x51\x70\xee\x07\x63\x5a\x37\xd6\xa9\x79\x69\x25\xbc\x84\x85\x71\x8c\xfa\x1f\x5a\x1b\xd4\x69\x77\x33\x24\xd2\xaf\xee\x22\x16\xf7\x57\xca\xd0\x71\xbb\x6a\x61\x98\x55\x93\xa6\x2b\x5c\xca\xec\xc3\xcb\x78\x04\x1b\x71\xce\x3d\x73\x45\xfb\x0b\x95\xf5\x20\x8b\x43\x11\xe9\xab\x6b\xe7\x12\xc2\x51\x29\x7c\x75\xc7\xff\0\xaa\x9d\x73\xa5\xcc\xb7\x8d\x71\x61\x73\xf6\x76\x97\xfd\x62\x95\xdc\x18\xfa\xe2\x8b\x87\x28\xb1\x68\xe0\x5a\xdc\xa4\xb2\x99\x27\xb9\x52\x1e\x52\x3a\x7d\x05\x5f\xb6\x8b\xc8\xb7\x8a\x2d\xd9\xf2\xd4\x2e\x71\xe8\x29\x5e\xe3\x4a\xc3\xc8\x15\x90\xba\x3c\xd1\xb4\xbe\x4e\xa1\x24\x69\x2c\x8c\xe5\x55\x07\x04\xf5\xe6\x84\xec\x0d\x5c\x90\xe8\xf0\xfd\x92\x3b\x65\x76\x08\xb2\x09\x1f\x3c\x99\x0f\xbd\x58\xd4\x6c\x92\xf6\xdf\xca\x66\x29\x86\x0c\xac\xbd\x41\x14\xef\xa8\x58\xb1\xdb\xd6\x8a\x92\x83\xe9\x49\x40\x09\x45\0\x27\x7a\x4a\x60\x21\xa4\x34\x80\x4a\x46\xa6\x01\x49\x40\x08\x69\x0d\0\x27\x7a\x4a\x04\x25\x21\x1c\xd3\x02\x7c\xd1\x9a\x62\x14\x52\xd2\0\xef\x4b\xf5\xa0\x02\x94\x50\x02\xd2\xd0\x02\x12\xa3\xae\x01\x34\x8c\xc9\x10\xcb\x10\xab\xef\xc5\0\x39\x4a\x3a\x82\xb8\x60\x69\x1e\x48\xe3\x03\x73\x05\xcf\x4c\xd0\x02\x79\xb1\x14\x32\x6f\x5d\xa3\xf8\xb3\xc5\x24\x32\x43\x29\xc4\x6e\xac\x47\xf7\x4d\x01\x71\xed\x24\x69\x22\x23\x32\x86\x7c\xed\x04\xf2\x71\xd7\x15\x2f\x18\xa4\x01\xda\x9d\x40\x0b\x43\x30\x8d\x4b\x31\xc2\xa8\xc9\x3e\x82\x80\x23\x86\xee\x19\x52\x36\x49\x01\x12\xfd\xcf\xf6\xaa\x73\xc7\x5a\x2c\x04\x73\xcd\x1d\xbc\x2f\x2c\xad\x84\x41\x92\x7a\xe0\x53\x92\x45\x91\x15\x94\xe4\x30\xc8\xa0\x07\x70\x0d\x2d\x20\x21\x9e\xea\x38\x66\x86\x27\x3f\x3c\xc4\x84\x18\xeb\x81\x93\x4b\x77\x75\x15\xa5\xbb\xcf\x3b\x6d\x8d\x31\x93\x8c\xf7\xc5\x3b\x05\xc7\xc9\x2a\xc7\x0b\x4a\xd9\xda\xaa\x58\xe0\x76\x14\xdb\x69\xe3\xb8\x82\x39\x63\x3f\x24\x8a\x18\x7d\x0d\x2b\x01\x21\x38\x34\xea\x06\x45\x6b\x70\xb7\x10\x2c\xa1\x59\x01\xcf\x0e\x30\x78\x3e\x95\x30\x20\xd0\x22\x17\xbb\x89\x2e\x56\xdc\xb7\xef\x1d\x4b\x01\x8e\xc2\x99\x61\x7d\x15\xf5\xb8\x96\x3d\xc1\x49\x23\x0d\xd7\x83\x8a\x76\x0b\x96\x0b\x0c\xf5\xa3\x83\xcd\x21\x94\x35\x8d\x52\x2d\x2e\x24\x79\x55\x9b\x7b\x60\x2a\xf5\xfa\xd5\xb4\x91\x5e\x15\x91\x5b\x2a\xc3\x20\xd3\xb6\x82\xbe\xb6\x2b\x35\xf4\x63\x51\x8e\xd0\x72\xce\x85\xc1\xcf\xa5\x59\xf3\x10\xb6\x37\x02\x45\x2b\x05\xca\xb0\x5f\xa4\xb7\x37\x10\xb0\xd8\x61\x60\xa4\x93\xd7\x23\x35\x65\xdd\x15\x37\x33\0\xa3\xb9\xa1\xa1\xdc\xaf\x7d\x78\xb6\xd6\x32\x5c\xa8\xf3\x02\x2e\x70\x0f\x5f\xc6\xb3\x9b\x57\xbd\x8a\x0f\xb4\x4d\xa6\xba\xc3\x8d\xc5\x84\x80\x90\x3d\x71\x54\x90\x9b\xb1\x3d\xde\xa9\xb4\x40\x96\x91\x19\xe7\xb8\x1b\x91\x01\xc7\x1e\xa4\xf6\xa6\x41\xa9\x5c\xc7\x79\x15\xbd\xfd\xaf\x92\x65\xfb\x8e\xaf\xb8\x13\xe9\x45\x82\xe4\x5f\xda\x97\xb2\xdd\x5c\x45\x6d\x64\x25\x58\x5f\x6e\xe3\x28\x5f\xe7\x53\xdb\x6a\xc8\xfa\x74\xb7\x33\x21\x89\xa0\x25\x24\x43\xc9\x0c\x3b\x51\x61\x73\x0c\x9f\x54\x9a\x33\x04\x11\xdb\x19\x6e\xe5\x4d\xe6\x35\x6c\x05\x1e\xe4\xd3\x5b\x5a\x09\x64\xf2\xbc\x2c\x26\x49\x04\x6d\x16\x79\xdc\x7b\x66\x8e\x50\xe6\x1d\x69\xa9\x4d\x25\xc9\xb5\xba\xb6\x30\x4c\x53\x7a\x7c\xdb\x83\x0a\xb1\xa6\x5e\x7d\xb2\xdc\xb3\x2e\xc9\x11\x8a\x48\xbf\xdd\x61\x43\x43\x4c\x7d\xd5\xdc\x16\xa0\x79\xf2\xa4\x79\xe9\xb8\xe3\x34\x8f\x79\x6c\x90\x09\xda\x68\xfc\xa3\xd1\xb7\x70\x69\x59\x8e\xe8\xaf\x7b\xa9\x22\xe9\x92\xdd\x5a\xbc\x72\xec\xe9\xce\x47\x5a\x92\x7d\x42\xda\xd9\x50\xcf\x32\xa1\x71\x90\x09\xa2\xc2\xb8\xb2\x5f\xdb\x47\x14\x72\x3c\xc8\x23\x90\xe1\x5b\x3c\x1a\x92\x19\x52\x68\x56\x58\x9b\x72\x30\xc8\x3e\xb4\x58\x77\x1f\x9a\x0f\x4a\x43\x12\x8e\xb4\0\x87\x14\x94\0\x86\x92\x98\x08\x69\x28\x01\x29\x28\0\x38\xa4\xa0\x42\x71\x49\x8a\0\x43\x49\x40\x13\xd1\x54\x21\x69\x7a\x0a\x40\x14\x50\x02\xd2\x8c\xd0\x02\xd1\x40\x18\xba\xda\xce\x97\xf6\x72\x19\x89\x89\xa7\x50\x23\x03\xa7\xe3\x46\xb2\x03\x6a\x76\xea\xf0\x9b\xa4\xf2\xcf\xee\x53\xa8\x39\xfb\xdf\xd2\xa9\x12\xc4\xd1\x2e\xa2\xb4\xb1\x94\xdc\x49\xe5\xa8\x99\x94\x46\x72\x7c\xbf\xf6\x69\x75\x55\x6b\xfb\xab\x59\x20\xb6\xfb\x54\x21\x18\xfd\xed\xa0\x93\xef\x47\x5b\x87\x4b\x11\x33\x2d\xd6\x92\x6d\xed\x2c\xf6\xba\xce\x16\x48\x81\xe3\x23\x93\xcf\xe1\x57\x34\xb7\x8a\x0b\xe7\xb5\x7b\x35\xb5\x98\xa6\xf1\xb4\xe4\x32\xfd\x69\x30\x43\xf5\x56\xf2\xf5\x5d\x3d\x89\xf9\x47\x98\x79\xff\0\x76\xb3\x4e\xb1\x70\xd1\x7d\xa9\x2e\x0b\x3e\x73\xf6\x61\x09\xc6\x33\xd3\x77\xd2\x84\x81\xbb\x32\xfe\xeb\xdb\xcd\x4a\xea\x08\xae\xbc\x88\xe2\x54\x23\xe4\x04\xe4\x8c\xd5\x73\xa8\xdd\x4d\x65\x69\x23\xbb\x45\x1b\x86\x12\xcb\x1a\x6e\x20\x83\x8e\x9d\xba\x51\x64\x17\x66\xae\x89\x39\x9e\xd4\xb3\x4e\x97\x18\x62\x03\xa8\xc1\xc7\xb8\xec\x6a\xdd\xe7\xfc\x7a\x4f\xff\0\x5c\xdb\xf9\x54\xb2\xba\x1c\xd4\x10\x3b\x9d\x10\xac\xcd\x18\x65\x6c\x60\x0e\x08\x19\x27\xf1\xad\xed\x61\xae\x13\x4f\x73\x68\x18\xc9\xc7\xdd\x19\x6c\x67\x9c\x0f\x5a\xa7\xb9\x2b\x63\x17\xed\x3e\x6d\x8d\xf4\x62\xea\x49\x07\x90\x49\x8a\x75\xc3\xa9\xf5\xf7\x15\x3c\x4b\x71\x64\xfa\x6b\xfd\xaa\x49\x04\xf8\x47\x8d\xb1\xb7\xee\xf6\x1d\xa9\x88\x6a\x3d\xd5\xc6\x99\x2e\xaa\x2f\x24\x8d\xd7\x73\x24\x63\x1b\0\x07\xa1\x1d\xea\xee\x9b\x77\x2c\xfa\xa4\xa1\x99\x82\x18\x23\x70\x84\xfd\xd2\x41\xa4\xc6\x8c\xf8\x26\x79\x1f\x4e\x96\x57\x67\x22\x79\xb0\xc4\xf3\x80\x0f\xf8\x54\x77\xd0\xc9\x73\xe1\xd9\x2f\xe5\xbb\x94\xc8\xfc\x94\xdd\xf2\x01\xbb\xa6\x29\x88\xe8\xae\xf9\xd3\x27\xed\xfb\x96\xeb\xfe\xe9\xae\x7a\x28\x9e\xd3\x46\xb3\xbe\x8a\xea\x56\x94\x14\x1b\x0b\x7c\x84\x1e\x36\xe2\xa5\x0d\x8a\xf2\x5d\xde\x49\x76\xe6\x3b\xb2\xc9\x23\x2c\x6d\x1c\x81\x55\x31\xd3\x23\xf9\xd7\x45\xa6\x34\xcf\x61\x0b\xdc\x63\xcc\x28\x37\x11\xdc\xd0\xf6\x04\x73\xb6\xcd\x2d\xcc\x7a\x54\x6f\x71\x30\xf3\x24\x98\x3b\x07\x39\x20\x13\xde\xb4\xf4\x90\x6d\xb5\x6b\xeb\x44\x91\xda\x14\x08\xc8\xac\xc5\xb6\xe4\x73\xc9\xa6\xc1\x11\xdf\xda\xc4\xfe\x25\xb4\x76\x2d\x93\x1b\x36\x37\x11\x82\x31\x59\xb0\x59\xa2\xe8\x37\x17\xa1\xa4\x13\xc6\xec\x63\x60\xe4\x6c\xc3\x76\x1f\x9d\x09\x89\xa2\x4b\xaf\xb4\x5f\xea\x73\xc6\xd6\xdf\x69\x8e\x34\x40\xab\xe7\x6c\x0b\x91\x9c\xfb\xd6\xb7\x87\x4c\xff\0\xd9\xdb\x6e\x18\x31\x46\x2a\xa4\x38\x6e\x3b\x64\x8e\xfd\xa9\x3d\x86\xb7\xb9\x9f\xa8\x19\xaf\x75\xb7\xf2\x2d\x4d\xd4\x76\xf1\x18\xd8\x6f\x0b\x82\xc3\x9e\x4f\xb7\x15\x59\x65\x91\xb4\x6b\x7b\x3b\x8d\xc8\x63\xba\x16\xf3\x60\xff\0\x0f\xf9\xc5\x3d\x04\x4d\x34\x10\x58\x6b\x2c\x74\xf5\xda\xe2\xd1\xdb\x60\xe7\x04\x74\xa8\x9e\x0b\x78\x74\x38\x2f\xad\xdb\xfd\x37\xe5\x22\x40\xc4\xb3\x31\x23\x20\xfa\xf7\xa0\x7a\x13\xdb\xd9\xd9\xdd\xeb\x1a\x97\xdb\x10\x3b\x2e\xcc\x02\x4f\0\xaf\x26\xa8\x5b\xfd\xa2\xe7\xec\x31\x3a\xa4\xd0\xed\x72\x8b\x33\x10\xaf\x83\x81\x9f\x5c\x0a\x7e\xa2\x27\x31\x3c\x5a\x7e\xad\x1e\xe8\x42\0\x08\x8a\x22\x48\x8c\xe3\x9e\xa3\xe9\x4e\xbd\xb2\xd4\x3f\xb1\x4c\x8f\x7f\xbe\x21\x10\x63\x1f\x96\x06\x47\xa6\x45\x17\x57\x0b\x32\x68\x25\x8e\xdb\x58\xb5\x9e\x40\x22\x86\x7b\x45\x48\xc9\x3c\x29\xeb\x8c\xd3\x2f\x6e\xee\x20\xd5\xad\xe2\x4b\xf1\x28\x96\x60\x0c\x42\x35\xf9\x54\x9f\x5a\x56\x1d\xec\x43\x04\x50\xb6\xa3\xa8\xf9\xb7\xef\x69\xfb\xdc\xe1\x64\x0b\x9e\x3d\xea\xba\x16\xfe\xc9\x9a\x20\x7c\xcb\x78\xee\xd7\xf7\xb8\xff\0\x58\x33\xc9\x3e\xbd\xb9\xa6\x23\x57\x55\x02\x3d\x52\x09\xa0\x9d\x20\xb9\xf2\xca\xe2\x55\x3b\x5d\x7e\xbe\xb5\x8f\x22\xbf\xd9\xae\xe4\x9a\x45\x92\x3b\x8b\x94\x43\x30\x1c\x2e\x3a\xb0\xfa\x74\xa4\x81\x9a\x16\x51\xa5\xbe\xbd\x0a\xc3\x70\xd7\x7e\x64\x44\x3b\x39\xdc\x63\xc7\x23\x9f\x7a\xb7\xa3\x02\x35\x1d\x4f\x6f\xfa\xbf\x35\x71\xf5\xc7\x34\x31\xa2\x29\x5a\x04\xf1\x0c\x86\xfb\x60\x5f\x24\x08\x8b\xf4\xf7\xeb\xde\xa9\x23\x5a\x2d\xed\x93\xaa\x14\xb2\x0f\x21\x5d\xff\0\x77\x77\x63\xf4\xf4\xa3\x50\xd0\x2f\x76\x49\xfd\xac\xf6\xa0\x79\x26\x25\x04\xaf\x42\xf9\xed\x53\xda\xcb\x0c\x1a\xb4\xcf\x7a\x42\x6f\x85\x3c\xa3\x27\x1f\x2e\x39\x1f\x9d\x3e\x80\x52\x8a\x10\xe2\xd8\x94\xff\0\x47\x92\xfc\x98\xd4\x8e\x36\xf3\xdb\xd2\xba\x90\xa1\x46\0\xc0\xf6\xa9\x90\xe2\x1d\xe8\xa9\x2c\x43\x45\0\x21\xa4\xa0\x04\xa4\x34\0\x86\x90\xd3\x01\x29\x28\x01\x28\x34\0\x9d\xa9\x38\xe9\x40\x84\x34\x86\x80\x26\x1d\x28\xaa\x10\xa3\xde\x96\x90\x0a\x68\xa0\x05\xa2\x80\x16\x81\x40\x15\xaf\xed\x0d\xd1\x80\xab\x05\xf2\xa5\x0f\xf5\xc5\x45\x79\x65\x39\xbb\x5b\xbb\x39\x11\x65\x0b\xb1\x83\x8e\x08\xfc\x29\xa6\x26\x89\x34\xfb\x1f\xb3\xac\xad\x3b\x09\x65\x99\xf7\xb9\xc7\x19\xf4\xa2\xf2\xd2\xe0\xcc\xb3\x59\xce\x23\x21\x76\x98\xdc\x12\x87\xdf\x8e\xf4\x5f\x50\xb6\x84\x29\xa4\xca\x96\xed\xb6\xe9\x96\xe1\xa5\xf3\x4c\x80\x70\x4f\xa1\x1e\x95\x2d\x9d\x8c\xab\x76\xd7\x57\x72\x89\x66\xdb\xb1\x70\xb8\x0a\x28\xb8\x58\x9e\xea\xcc\x5c\x5d\x5b\xcc\xcd\x81\x0e\xef\x97\x1d\x72\x31\x55\x23\xd2\xa6\x8d\x44\x09\x78\xc2\xd4\x36\x76\x05\xc3\x63\x39\xc6\xef\x4a\x57\x0b\x17\xad\xed\x04\x37\x97\x17\x01\x8e\x66\xdb\x91\xe9\x81\x8a\xaa\x9a\x4b\xc1\x04\x2b\x6b\x74\xd1\xbc\x59\xe7\x6e\x55\x81\x39\xe4\x51\x70\xb1\x2d\xb6\x98\xd0\x43\xb5\x2e\x18\x3b\x4d\xe6\xc8\xe0\x01\xb8\xf7\x18\xec\x2a\xfb\xa0\x92\x26\x43\xd1\x94\x8f\xd2\x86\xee\x34\x8c\xf6\xd2\x14\x45\x68\xa9\x33\x23\xda\x9f\x95\x80\xea\x0f\x51\x56\xaf\xed\x16\xf2\xdf\xca\x2e\xd1\x90\x43\x2b\xaf\x55\x23\xa1\xa2\xe2\xb1\x55\x34\x75\x71\x21\xbb\x9d\xe7\x96\x48\xfc\xad\xf8\x03\x6a\xf5\xe0\x0a\xb5\x25\x8c\x72\x7d\x9b\x71\x6c\x5b\x9c\xaf\xbf\x18\xe6\x8b\x85\x8a\x9f\xd8\x71\x7c\xf1\xac\xf2\x8b\x67\x6d\xc6\x01\x8d\xbd\x73\x8c\xf5\xc5\x4f\x75\xa5\xa4\xd7\x2b\x3c\x73\x4b\x6f\x26\xdd\x8d\xe5\xe0\x6e\x5a\x2e\x16\x0b\x7d\x1e\xda\x01\x6e\xa8\x64\xc5\xbb\xb3\xa6\x4f\xf7\xba\xe6\xa1\x97\xc3\xf6\xd2\x23\xc5\xe7\x4e\xb0\x31\xdc\x22\x0f\xf2\xa9\xce\x78\xe2\x8e\x60\xe5\x46\x94\x91\xac\x90\xb4\x4d\xf7\x59\x4a\x9f\xa6\x2b\x3e\xdb\x42\xb6\x81\xe3\x22\x49\x9e\x38\xce\xe5\x89\xdf\x2a\x0f\xae\x29\x5c\x6d\x5c\x92\x7d\x1a\xda\x69\x9d\xf7\xcb\x18\x90\xe6\x48\xe3\x7d\xaa\xe7\xd4\x8a\xbf\x1c\x6b\x1c\x6a\x88\xa1\x55\x46\0\xf4\x14\x5c\x12\xb1\x56\x1d\x32\xda\x2f\x23\x62\xb0\xf2\x0b\x14\xe7\xa6\xee\xb5\x32\x5b\x44\x97\x12\x5c\x2a\xe2\x49\x40\x0c\x7d\x71\xd2\x8b\x85\x86\x5d\x58\xc3\x73\x34\x32\xc8\x18\x49\x09\xca\xb2\x9c\x7e\x06\x91\x74\xfb\x65\xb3\x7b\x5d\x87\xc9\x72\x49\x5c\xfa\x9c\xf5\xa2\xe1\x61\x97\x5a\x5d\xad\xc3\xab\xb8\x65\x75\x5d\xbb\x91\xc8\x24\x7a\x1a\xb1\x6f\x04\x56\xd1\x2c\x50\xa0\x44\x51\xc2\x8a\x2e\x16\x08\x2d\xa2\xb7\x2e\x62\x5d\xa6\x46\x2e\xc7\x3d\x4d\x45\x26\x9f\x6b\x2f\x9d\xbe\x20\xde\x7e\x3c\xcc\xe7\xe6\xc7\x4a\x2e\x16\x1b\x6f\xa6\x59\xdb\x4a\x24\x86\x1c\x48\x38\xdd\xb8\x93\xfa\xd4\x71\x69\x16\x31\x4e\x25\x8e\xdd\x43\x83\xb8\x72\x70\x0f\xb0\xe8\x28\xbb\x0b\x21\x6e\x74\xab\x1b\x99\x4c\x93\x5b\xab\x39\xea\xd9\x23\x3f\x91\xa7\xcf\x61\x6b\x71\x6e\x90\x4b\x0a\xb4\x69\xf7\x57\xa6\x28\xbb\x0b\x20\x8a\xc6\xda\x2b\x73\x6f\x1c\x2a\x22\x6c\xe5\x7d\x7e\xb5\x23\x41\x1b\x40\x61\x64\x06\x3d\xbb\x76\x91\xc6\x28\xb8\xec\x47\x25\x9d\xb4\xb0\x2c\x12\x42\x8d\x1a\xf0\x14\x8e\x94\xc8\x34\xdb\x2b\x70\x04\x56\xd1\xae\x18\x30\x38\xe7\x23\xde\x8b\xb0\xb2\x16\x4d\x3e\xce\x59\x0b\xc9\x6b\x13\x39\xea\xc5\x01\xcd\x13\xd9\xc1\x35\xa3\xda\x94\x0b\x13\x0c\x6d\x50\x06\x3d\xc5\x17\x61\x64\x30\x59\x24\xd6\xb1\x45\x7c\x89\x3b\xa0\xc1\x62\x3b\xd4\xa2\xde\x15\x83\xc8\x11\x27\x94\x06\x36\x63\x8c\x51\x70\xb0\xdb\x7b\x38\x2d\x43\x7d\x9a\x14\x8c\x9e\xa4\x0a\x6d\x85\xaa\xda\x5b\x94\xdd\xbd\xd9\x8b\xbb\x7f\x79\x8f\x5a\x2e\x16\x1f\x35\xbc\x37\x18\x13\xc4\x92\x01\xd3\x72\x83\x8a\x1e\x08\x9e\x3f\x2d\xa2\x43\x18\xfe\x12\xa3\x1f\x95\0\x0b\x0c\x49\x1f\x96\x91\xa2\xa7\xf7\x40\xc0\xa6\xcb\x04\x33\x63\xcd\x89\x24\xdb\xd3\x72\x83\x8a\0\x7e\xc5\x20\x0d\xa3\x0b\xd3\x8e\x94\x1e\x29\0\x99\xa0\xd0\x31\x29\x29\x80\x77\xa4\xa0\x04\xed\xc5\x25\0\x21\xe9\x49\x40\x08\x79\xa4\xa0\x03\x8a\x4a\x04\x25\x21\xa0\x04\xa4\x22\x80\x25\xa5\xa6\x21\x68\xef\x4c\x05\x14\xb4\x86\x1d\x29\x7e\xb4\0\x52\x8a\0\x6b\x4d\x1a\x38\x56\x75\x04\xf4\x04\xf2\x68\x92\x68\xe3\xce\xf7\x55\xc7\x5c\x9a\x04\x2b\xcb\x1a\x46\x64\x77\x55\x4f\x52\x78\xa2\x29\xe2\x95\x37\xc6\xea\xca\x3b\x83\x9a\0\x48\x6e\xa0\x99\x8a\xc5\x2a\x39\x5e\xa1\x58\x1c\x50\xb7\x96\xed\x2f\x94\x93\x46\xcf\xfd\xd0\xc3\x34\x59\x85\xd1\x38\xa5\x1e\xf4\x86\x2d\x2f\x34\0\xa2\x9c\x33\x48\x02\x94\x53\0\xed\x4b\x9c\x52\0\xa5\x14\0\x80\xe4\x52\xd0\x02\x52\xf6\xa0\x05\xa3\xbd\0\x03\xeb\x45\0\x2d\x25\0\x07\xad\x02\x80\x0f\x6a\x08\xcd\0\x26\x68\xcd\0\x26\x79\xe6\x90\x50\x01\x9e\x68\x3d\x28\x01\x28\xa0\x02\x93\xf0\xa0\x02\x92\x81\x85\x25\0\x27\x4a\x4a\0\x29\x28\0\x38\xa4\xa0\x04\xfa\x52\x53\0\x14\x50\x02\x52\x52\x01\x29\x31\x4c\x06\xf7\xa2\x80\x0f\xc2\x90\xd0\x02\x1a\x4a\0\x43\x48\x7d\xe8\x10\x94\x9d\x68\x02\x5a\x70\xaa\x10\x52\xd2\0\xcd\x14\0\xb9\xa2\x80\x16\x81\xd6\x80\x39\xdb\xd8\x57\xce\xbc\x93\x64\x57\x28\x18\x97\xcb\x6d\x92\x3f\x61\x9a\x96\xdc\x5b\x5c\xeb\x04\xdc\xa8\x21\xad\xd0\xaa\xc9\xfa\xfe\x35\x7d\x08\x2b\xa6\xd3\x6d\x1c\x0a\x82\x44\x7b\x97\xf2\x37\xb1\x08\x14\x7a\xfa\xf7\xa6\x21\x74\x83\x53\x44\x28\x3e\xe6\x44\x39\xda\x07\x7c\x53\x11\x7e\xe9\x6c\xd3\x06\xc9\x40\xb8\xf2\x1f\x6f\x95\xd0\x8c\x77\xfe\x95\x42\x18\x8b\xc5\x65\x89\x6d\x51\xb7\x21\x5f\x29\x4e\xff\0\xc7\xfa\xe6\x92\xf3\x07\xe4\x75\x50\xcd\x1c\x8c\xca\x8e\x19\x90\xe1\x80\x3d\x0d\x4a\x38\x15\x99\xa8\xa3\xda\x94\x1a\x04\x38\x0e\x69\x72\x05\x03\x0c\xe2\x8c\xf1\x48\x05\xa4\x06\x80\x17\x34\x7b\xd3\x01\x68\xa4\0\x29\x73\x40\0\xa0\x9f\x6a\0\x51\xd2\x8e\x28\x01\x3b\xd2\xf1\x40\x09\x9c\xd1\xdb\xad\0\x14\x1a\0\x41\x41\xf5\xa0\x04\xf6\xa3\x39\x3c\x50\x01\x45\0\x27\x7a\x4c\xd0\x01\xf5\xa2\x80\x10\x9a\x3b\x50\x30\xa4\x34\x08\x0d\x21\xe2\x80\x13\xe9\x49\x40\xc0\xd2\x1a\0\x4a\x40\x69\x80\x50\x4d\x20\x1a\x4d\x14\xc0\x43\x8a\x6d\0\x14\x84\xf3\xeb\x40\x09\x49\x40\x05\x27\x14\0\x84\xd2\x1a\x04\x21\xa4\xa0\x09\x46\x3d\x68\xe9\xd2\xa8\x42\xd2\xd2\0\xeb\x4b\x40\x07\x5a\x51\xd6\x80\x0e\xbd\x29\x45\x03\x2b\xcb\x61\x6b\x3c\xc2\x59\x60\x46\x7f\x5f\x5f\xad\x43\x26\x97\x1c\xfa\x83\xcf\x3c\x71\xbc\x6c\x8a\xa0\x11\xc8\x22\x9a\x64\xd9\x16\xa4\xb3\xb7\x96\x11\x0c\x90\xa1\x8c\x74\x5c\x70\x3e\x94\x45\x67\x6d\x11\x26\x38\x11\x49\x5d\xbc\x0e\xa3\xd2\x95\xc7\x61\x6d\xac\xed\xad\xd9\x8c\x10\xa2\x16\xea\x54\x51\x1d\x9d\xb4\x72\x99\x63\x82\x35\x73\xfc\x41\x46\x68\xb8\x58\x99\x23\x44\x2c\x55\x55\x4b\x72\x48\x18\xc9\xa7\x83\xe9\x48\x05\x27\xde\x95\x68\x18\xf1\xd2\x8a\0\x0f\xeb\x40\xa4\x01\x40\x3e\x94\xc0\x5e\x69\x73\x48\x02\x8c\xd0\0\x4f\x34\xbd\xa8\0\xe9\x45\0\x2d\x1d\xba\xd0\x01\x40\xc5\0\x27\xb5\x14\0\x7e\x34\x13\x40\x06\x69\x28\0\xef\x49\x40\x0b\xd3\xa5\x27\x6e\x68\0\xce\x69\x28\x01\x0d\x14\x0c\x28\xa0\x03\xb5\x25\0\x25\x27\x14\0\x52\x1a\0\x4a\x4a\x60\x04\x50\x29\0\x9d\xe9\x0d\x30\x10\xd2\x1a\x40\x21\xa2\x98\x84\xa4\xa0\x60\x78\xa4\xa0\x04\xa4\x34\0\x7d\x69\xbd\xa8\x10\x9d\x69\x28\x02\x41\x4b\x54\x21\x69\x69\0\x66\x97\xbd\0\x19\xf7\xa5\xa0\x02\x94\x50\x02\xd0\x28\x18\x67\x9a\x5a\x04\x28\xeb\x4b\x9f\x6a\x43\x0a\x50\x68\0\xcd\x3d\x68\x01\xfd\xfa\x50\x4d\0\x19\xa4\xe8\x68\0\xcd\x2e\x69\0\x66\x97\x3c\x53\0\xc8\xf5\xa2\x90\x0b\xc0\xa5\xa0\x04\xfa\x52\xd0\x01\xd6\x8c\xf3\x40\x05\x26\x78\xa0\x03\x34\xbf\x5a\0\x4c\xd1\x9c\xd0\x02\x66\x8e\xb4\0\x74\xa2\x80\x0a\x4c\xd0\x02\x71\x40\xa0\x02\x8e\xbd\x68\x18\x1a\x4c\xd0\x01\x9a\x4a\0\x3b\x52\x1a\0\x0d\x37\xbd\0\x14\x94\0\x94\x50\x02\x51\x4c\x04\xeb\x49\x48\x02\x92\x80\x1b\x47\x7a\x60\x21\xf5\xa0\xd0\x02\x1a\x43\x40\x09\xde\x90\x8a\x04\x36\x8a\0\x92\x97\x35\x42\x16\x8a\0\x29\x73\x48\0\x75\xa5\x06\x80\x14\x73\xda\x8a\0\x28\xa0\x05\xa5\x14\0\xb9\xa2\x81\x8b\x9a\x33\xda\x80\x14\x72\x70\x2a\x51\xd2\x90\x85\xcd\x19\xe2\x81\x89\x46\x68\0\xcd\x14\0\xb9\xa5\x34\0\x62\x80\x69\0\xbc\xe6\x8e\xf4\xc0\x5a\x4a\x40\x2d\x14\0\xb4\x94\0\x64\x52\x50\x01\x9a\x38\xa0\x02\x93\x3e\xb4\0\x66\x8c\xd0\x01\x48\x68\0\xfc\x28\xa0\x05\xa4\x3e\xd4\x0c\x43\xd3\x9a\x33\x40\x09\x47\x4a\0\x43\x45\0\x21\xa4\xef\x40\x07\x6a\x6d\0\x14\x50\x02\x52\x53\x01\x28\xa0\0\x7b\xd2\x52\x01\x29\x3b\xd3\x01\x0d\x07\xde\x80\x12\x92\x81\x09\x4d\x3e\xd4\0\x94\x94\x01\x2f\xe3\x46\x6a\x84\x2f\xd6\x8a\0\x5a\x28\0\x14\xbe\xd4\0\xa2\x97\x34\x80\x81\x2e\x62\x6b\xa9\x2d\xd4\x9f\x31\0\x24\x63\xb1\xa0\x5d\x44\x6e\xda\xd8\x13\xe6\x2a\xee\x3c\x76\xa7\x60\xb9\x3e\xe1\xd3\x34\xb9\x18\xeb\x48\x61\xb8\x0e\xa4\x53\xb3\xf9\x50\x21\x37\x0c\xf5\xc5\x28\xe7\x9a\x06\x3d\x0a\xfa\xd3\x2d\xee\x04\xc6\x40\x51\xd3\xcb\x72\xbf\x30\xc6\xef\x71\xed\x48\x09\x77\x29\xe8\x69\x03\x02\x78\x34\0\xa4\x80\x32\x4d\x56\x4b\xd8\xde\xf9\xed\x80\x3b\x95\x03\xe7\xb1\x04\xd0\x22\xc0\x91\x49\xc0\x23\x3f\x5a\x50\xc3\x19\xcf\x14\x0c\x78\xe7\xa5\x1d\xe8\0\xa3\xe9\x40\x0a\x38\xcd\x28\xeb\x40\0\x34\x67\xd2\x90\x07\x34\xb4\xc0\x33\x49\xf5\xa0\x02\x8c\xd2\x01\x09\xfd\x68\xf7\xa0\x02\x92\x80\x0f\xa5\x19\xa0\x03\xbd\x19\xa0\x03\xf1\xa3\x34\x0c\x29\x28\x01\x0f\x34\x0a\0\x28\x3c\x50\x03\x4d\x1d\xf9\xa0\0\xd2\x1c\x53\x01\x0d\x25\0\x21\xcd\x14\0\x52\x50\x02\x75\xa3\xb7\x34\0\x1a\x43\x8a\0\x43\x49\x40\x09\x49\x40\x05\x34\x9a\0\x4a\x4e\x94\xc4\x27\xd6\x90\xf4\xa0\x09\x29\x45\x31\x0a\x68\xa0\x02\x96\x90\x0a\x28\x1e\xd4\xc0\x51\x46\x69\x01\x89\x20\xba\x3a\xf5\xc0\xb4\x68\x83\x18\xd7\x26\x40\x7f\xa5\x46\x1a\x78\x35\x3b\xa7\xba\x92\x35\x90\x5b\xe7\x7a\x03\x80\x3f\x1a\xa2\x35\x22\x81\xe5\x4b\xbb\x37\x1f\x69\x51\x2b\x15\x76\x95\xb8\x7e\x3d\x33\x52\x99\xa7\x11\xb6\x9b\xe6\x37\x9c\x67\xda\x18\xb7\xcd\xb3\xef\x67\xf2\xa6\x22\x39\x1a\xea\xe6\xe2\xeb\x31\x4c\xde\x53\x94\x52\x92\xed\x09\x81\xfa\xd6\xde\x94\xd2\xbe\x9f\x13\x5c\x10\x64\xdb\xc9\x07\x39\xf7\xcd\x27\xb1\x4a\xf7\x30\x6f\x9d\x9a\xde\xe2\xee\x05\x9e\x42\xaf\x95\xb8\x67\xda\x07\x3d\x97\x3c\x8a\xe9\x2e\x1b\x1a\x7c\xcd\x9c\x1f\x2d\x8e\x7f\x0a\x18\x23\x15\x2d\x84\x16\x1a\x7d\xda\x49\x20\x9d\xe4\x8c\x33\x16\x27\x70\x6e\xd8\xa0\xcb\x26\x66\x8b\x7b\x2a\x4b\x7e\x51\x98\x1c\x60\x71\xc7\xe3\x4c\x5b\x12\x6a\x68\xba\x74\x33\x0d\x3e\x72\xa5\xd5\x73\x10\x6f\xbb\xce\x37\x64\xf4\xa6\xd8\x41\x35\xb6\xa7\x6f\xb6\x11\x6e\xaf\x9d\xe0\xce\x1f\xcc\x18\xf4\xf5\x14\xba\x07\x52\xf6\xb7\x99\x26\xb3\xb6\x72\x56\x09\xa4\x22\x42\x0e\x33\x81\xc0\xfc\x6a\x88\x8e\xda\xc3\x53\xbd\x30\x96\x48\x96\xdb\x24\x29\xc9\x4c\x9e\xdf\xcf\xf1\xa1\x6c\x37\xb9\x0d\xb4\x46\x0b\xeb\x06\x16\xe2\xdf\xcc\x62\x09\x32\x6e\x69\x06\x3b\x8a\x91\x87\x97\xbf\x46\x07\x97\xb8\x18\xc7\xfc\xf3\x3f\x31\xfe\xb4\xc4\x74\x88\xcb\x92\x80\x8c\xaf\x51\xe9\x4f\xac\xcd\x02\x97\x3d\xa8\0\xa5\xfa\x50\x01\xd6\x94\x52\0\x14\x7e\x34\0\x51\x9a\x60\x19\xa4\xcd\x20\x0c\xf3\x41\xa0\x04\xc5\x04\xd3\0\xa4\xfc\x29\0\x67\x14\x66\x80\x16\x8a\0\x4c\x66\x93\x34\x0c\x33\xe9\x45\0\x07\xad\x27\x7a\0\x4c\xd2\x1f\x6a\x60\x18\xa4\x3d\x28\x01\x0f\xa5\x04\x50\x02\x13\x49\xda\x80\x0a\x28\0\xa4\x26\x80\x03\xd6\x9b\x40\x07\x5a\x43\x40\x08\x69\x0d\0\x25\x21\xa0\x42\x52\x50\0\x69\xb4\x01\x20\xa5\xce\x2a\x84\x19\xa2\x80\x16\x96\x90\x05\x14\0\xb9\xa7\x50\x04\x0b\x6a\x8b\x74\xd7\x20\x9d\xec\xa1\x4f\xa7\x14\xd9\x2c\x20\x96\x69\x25\x75\x24\xc8\x9b\x08\xcf\x6a\x77\x15\x8a\xc3\x47\x84\x18\xc9\x96\x66\x68\xc8\x28\x59\xf3\xb7\xda\xad\xfd\x8a\x1f\xb6\x0b\xb2\xbf\xbd\x0b\xb7\x3e\xd4\x5c\x14\x51\x1d\xc6\x99\x6f\x3c\xa6\x42\x64\x42\xdc\x38\x46\xc0\x7f\xad\x5c\x8a\x34\x8e\x35\x8e\x30\x15\x54\x60\x01\x45\xc7\x62\x93\xe8\xd6\x6e\x5c\x32\xbe\xc6\xcf\xc9\xbc\xed\x52\x7b\x81\x57\x56\x25\x10\xf9\x27\x25\x36\xed\xe4\xf3\x8f\xad\x0d\xdc\x12\x48\x69\xb3\x81\xa1\x8a\x12\x9f\x24\x44\x14\x19\xe8\x47\x4a\x6b\x69\xf6\xcf\x1c\xb1\xbc\x40\xac\xaf\xbd\x86\x4f\x5f\x5f\x6a\x57\x0b\x09\x0e\x9b\x6b\x12\x48\x82\x32\xfe\x68\xc3\x97\x62\xc5\x87\xd4\xd2\xdb\x69\xd6\xd6\xb2\x79\x91\x21\xdd\x8c\x02\xcc\x5b\x03\xd0\x66\x8b\x8a\xc8\x9a\xe6\xde\x2b\xa8\xbc\xb9\xd0\x3a\xf5\xc7\xa1\xf5\xa8\x62\xd2\xec\xe1\x0c\x12\x11\x97\x5d\xac\x49\x27\x23\xdf\x34\x5d\x8e\xc8\x64\x7a\x35\x8a\x6d\x22\x1e\x55\x83\x29\x2c\x49\x18\xe9\x8f\x6a\x6d\xb5\x9c\x8d\xaa\x49\x7b\x70\xa8\xa4\x2f\x97\x18\x53\x9c\x8f\x53\x45\xc5\x62\xea\x41\x1a\xce\xf3\x2a\x01\x24\x98\xdc\xde\xb8\xe9\x52\x0f\x6a\x43\x17\x34\x7d\x68\x18\x52\x8e\x94\0\xb4\x99\xcd\0\x28\xeb\x47\xe3\x48\x02\x92\x80\x0e\xd4\x7d\x69\x80\x51\xda\x90\x01\x3e\xf4\x94\xc0\x28\xcd\x20\x12\x94\x74\xe6\x80\x0a\x3a\x50\x02\x1a\x28\x18\x76\xa4\x38\xc5\0\x07\x9a\x43\x4c\x42\x52\x13\xc5\x03\x0f\xc2\x90\xf1\x40\x05\x21\xa0\x02\x92\x80\x0a\x28\0\xfa\xd2\x50\x02\x52\x66\x80\x03\x48\x7a\x50\x02\x52\x50\x21\x29\xa6\x80\x0a\x4a\0\x4a\x43\x40\x12\x76\xa2\xa8\x42\xd1\x40\x0b\xed\x45\x20\x17\x22\x97\xa9\xa0\x05\x14\x50\0\x29\xc7\xd2\x80\x13\x14\xa2\x80\x16\x94\x7b\x50\x02\xd0\x0d\x03\x16\x96\x90\x80\x7d\x69\x73\xcd\0\x28\xa3\x34\x0c\x3a\xd2\x8a\x04\x14\x50\x31\x69\x33\x40\x0b\xde\x8f\xa5\0\x2f\xb5\x26\x45\0\x2e\x68\xa4\x01\x9a\x33\x9a\0\x4e\xd4\x53\0\xcd\x14\x80\x3b\x52\x74\xa0\x02\x8a\0\x28\x34\0\x03\x47\xd6\x81\x81\xa4\xa0\x41\x9a\x4a\x63\x0a\x42\x69\0\x77\xa4\xa6\x01\x49\xc5\0\x19\xa4\x34\0\x7d\x29\x28\0\xe8\x28\xa0\x04\xe7\xd6\x8a\0\x42\x45\x26\x7d\x68\x01\x09\xa4\xcf\xbd\0\x21\xa4\xe9\x40\x05\x27\x6a\x04\x27\x7a\x0d\0\x37\xe9\x45\x30\x1d\xda\x8a\x62\x14\xd2\xd0\x01\x4b\x48\0\x52\xf4\xa0\x05\xa5\xa0\0\x51\x40\x0a\x05\x2d\0\x28\xa5\x14\x0c\x29\x45\x02\x16\x96\x90\0\xa5\xa0\x04\xcd\x2f\x5a\x06\x2f\x14\xbd\xa8\0\xef\x41\xc5\0\x14\x50\0\x0d\x28\xa0\x02\x8a\0\x3b\xd1\x9a\0\x28\xa0\x02\x8c\xf6\xa4\x01\x49\x40\x07\x6a\x0f\x4a\0\x0f\x5a\x29\x80\x52\x0a\x40\x2d\x26\x69\x80\x13\x49\x9e\x69\x0c\x28\xc5\0\x21\xeb\x49\x4c\x03\x38\xa3\x39\xa0\x04\xa4\x34\0\x51\x8a\0\x29\x28\0\xa4\x34\0\x52\x50\x02\x52\x1e\x94\0\x86\x93\xf0\xa0\x41\x49\x40\x01\xa6\xe7\x8a\0\x3b\x52\x1e\x94\0\x94\x94\0\xff\0\xc6\x94\x55\x08\x29\x68\0\xa5\x14\0\xa2\x96\x90\0\xf6\xa5\x14\0\xa3\x8a\x0f\x4a\0\x05\x2d\0\x28\xa3\xbd\x03\x17\xbd\x2d\0\x2f\x5a\x3a\x52\0\xf6\xa0\x50\x02\xd2\x8a\0\x51\xcd\x14\0\x99\xa0\x7e\xb4\0\xb4\x50\x01\x4b\x40\x06\x68\xcd\0\x1f\x4a\x28\0\xa0\xd0\x01\x49\x40\x0b\x9e\x29\x3b\x74\xa0\x02\x8e\xbd\x68\0\x14\x1e\x69\0\x7d\x29\x29\x80\x51\x48\x04\xcd\x14\x0c\x0d\x29\x34\xc0\x4a\x42\x7f\x3a\x40\x27\x7a\x29\x80\x76\xa4\xfa\x50\x01\x46\x68\x01\x3b\xd1\xf4\xa4\x02\x51\x9a\x60\x21\xe9\x46\x68\x01\xa4\xd2\x1a\x04\x25\x14\0\x1a\x69\x34\0\x76\xa4\xa0\x04\xa4\xa0\x04\xc7\x34\x1a\0\x7d\x2d\x50\x83\xeb\x40\xe9\x40\x0a\x29\x45\0\x2d\x2d\x20\x01\xd3\x34\xb4\0\xa0\xd0\x4d\0\x02\x96\x80\x0e\x94\xa2\x80\x01\x4e\x14\x0c\x5a\x33\xcd\x20\x13\x34\xa2\x80\x17\x14\xb4\0\x66\x8a\x04\x14\x50\x31\x41\xa3\x34\0\x51\xd2\x80\x0c\x8c\x51\x9e\x68\0\xfa\x52\xd0\x02\x77\xa3\x34\0\x51\x40\x05\x06\x80\x13\x34\x67\x34\0\x51\x40\x01\xa3\xbd\0\x25\x14\0\x1a\x4c\xd0\x30\xa5\xcd\0\x27\x7a\x43\xcd\x02\x0a\x4c\xfa\x50\x01\x49\x40\xc5\xf5\xa6\xf6\xa0\x05\xfa\x52\x62\x80\x13\x34\x50\x21\x0d\x25\x03\x02\x79\xa6\xe6\x81\x05\x25\0\x14\x94\0\x9f\x4a\x28\x01\x33\x49\x4c\x02\x92\x80\x1f\x45\x31\x0b\xd2\x96\x80\x0a\x51\x40\x07\x7a\x5a\0\x5e\xd4\xa2\x90\x05\x21\xa0\x05\x14\xbf\x4a\0\x05\x3a\x81\x87\x14\xb4\0\xb8\xa3\xf1\xa4\x02\x8a\x29\x80\xbd\x68\xdd\x48\x41\xd2\x82\x68\x18\x66\x81\x40\x0b\x46\x79\xa0\x02\x8a\0\x33\x45\0\x1f\x85\x2d\0\x25\x14\0\x51\x40\x0b\x41\xe2\x80\x12\x92\x80\x01\x4b\xde\x80\x13\x3c\xd1\x9a\0\x29\x28\0\xa3\xa5\0\x14\x99\xfa\xd0\x30\x34\x94\0\x52\x7d\x68\0\xa3\x34\0\x12\x69\x33\x40\0\xa4\xcd\0\x1e\x94\x66\x80\x1a\x69\x09\xa0\x42\x13\x49\x9a\x60\x06\x93\xb5\x20\x0c\xfa\xd2\x50\x02\x66\x8f\xe5\x4c\x04\xa3\xbd\x20\x10\x9a\x4a\x60\x3e\x96\x98\x82\x9c\x28\0\xa5\xa0\x05\x1c\x51\x48\x05\x07\xda\x8e\xf4\0\x51\x40\x0b\xf5\xa5\x1d\x28\x18\xa2\x8a\0\x51\x4b\x9a\0\x29\x45\x21\x0b\x46\x28\x18\x76\xc5\x03\x14\0\x1f\xad\x14\0\x77\xa0\x1a\0\x5c\xd1\xd2\x80\x12\x96\x80\x0a\x3b\xf3\x40\x05\x2f\x6a\0\x29\x0f\xa5\0\x2f\x7a\x28\x01\x33\x41\x38\xa0\x03\xb5\x1c\xd0\x01\xda\x8c\xfa\xd0\x02\x74\xa3\xb5\0\x19\xa2\x80\x10\x9a\x28\0\xef\x49\x9a\x06\x1d\x68\xa0\x02\x92\x80\x0a\x42\x79\xa0\x02\x8c\x8a\0\x4f\x7a\x4c\xd0\x01\xd2\x92\x81\x01\xeb\x4d\x34\0\x94\x9d\x28\x01\x73\x48\x68\x01\xa4\xd1\x40\x05\x25\0\x14\x9f\x4a\x60\x25\x14\x01\x25\x14\xc4\x14\xa0\x8a\0\x5e\xf4\xa2\x80\x0c\xe2\x80\x79\xa4\x03\x87\xb5\x25\0\x19\xa5\xcd\0\x14\xe1\x40\x0b\x45\x03\x17\x34\xb4\0\x52\xe7\x23\x14\0\x1a\x4a\x40\x2e\x68\xfa\x53\x01\x7a\x50\x38\xa4\x01\x49\xd2\x80\x17\x3e\x94\x66\x80\x0a\x28\0\x1c\x51\x9a\0\x5a\x28\0\xa2\x80\x0e\xf4\x03\xef\x40\x06\x69\x0d\0\x14\x12\x7a\x50\x02\xd2\x1a\0\x4c\xd1\x9f\x5a\0\x4f\xad\x14\0\x51\x9e\x68\0\xcd\x25\x03\x0a\x43\xd6\x80\x17\x3e\xb4\x50\x02\x13\xe9\x47\xb5\0\x21\xa4\x34\0\x1a\x4a\0\x3e\x94\x9d\xe8\x01\x33\x48\x4d\x02\x10\x9a\x4a\0\x5e\x82\x92\x98\x09\xf4\xa2\x90\x09\x9a\x29\x80\x94\x99\xa0\x03\x34\x99\xa0\x09\x29\x7b\x53\x10\x52\x8a\0\x5a\x28\x01\x41\xa0\x50\x02\xe7\xd2\x8c\xd2\0\x14\xb4\0\x0a\x51\x40\x0a\x39\xa5\xa0\x05\x14\x74\xa0\x61\x4b\x40\x05\x14\0\x74\xa2\x80\x0c\xf3\x46\x68\x01\x69\x29\0\xb4\x74\xa0\x02\x96\x80\x10\x9e\x3d\xe8\xa0\x05\xa3\x34\0\x0e\x7b\xd1\xd2\x80\x0a\x39\xa0\x03\xa5\x14\0\x84\xfe\x34\x66\x80\x03\xde\x93\x39\xa0\x02\x8e\xb4\0\x1f\xca\x82\x7d\x28\x01\x33\xef\x41\xe2\x80\x13\x34\x13\xd2\x80\x17\xeb\x49\x9f\x6a\0\x29\x28\x18\x52\x7d\x68\0\xa0\xd0\x02\x52\x50\x01\x9a\x4a\x04\x21\x34\x84\xd3\x01\x38\xa3\x34\0\x66\x9b\x9f\x4a\0\x29\x28\0\x3c\x51\x9a\0\x43\xd3\x9a\x05\0\x26\x68\xcd\0\x49\x40\xa6\x21\x69\x45\0\x14\xa2\x80\x03\x4b\x48\x03\x34\xb4\0\x50\x28\x01\xd4\x50\x02\xd1\x9a\0\x51\xd2\x94\x50\x02\xfd\x28\x34\x0c\x4a\x5c\x50\x01\xd2\x8c\xe6\x80\x0a\x3b\x52\0\x14\x76\xa6\0\x68\xa0\x05\xa3\xf1\xa4\0\x05\x19\xa0\x02\x8a\0\x5c\xd1\x9a\0\x4a\x33\x40\x05\x1d\xa8\x01\x33\x45\0\x06\x93\xbd\0\x04\xd1\x9e\x68\0\xcd\x26\x7d\x28\x18\x51\x40\x06\x69\x28\x10\x13\x45\0\x06\x92\x80\x03\xfe\x4d\x1f\x4a\0\x4c\xf1\x8a\x43\x40\x06\x69\x09\xa0\x02\x9a\x7a\x50\x01\xda\x90\xd0\x02\x51\x40\x06\x69\x3b\x53\x01\x0f\x5a\x28\x01\x0d\x19\xa0\x04\xcd\x14\0\x84\xd0\x4d\0\x4b\x40\xa6\x21\x68\xa0\0\x52\x8a\0\x5a\x28\x01\x69\x7a\x76\xa0\0\x50\x29\0\xbf\xca\x8a\0\x53\x4a\x3a\x50\x31\x47\xbd\x2d\0\x04\xd2\xe6\x80\x13\x8a\x5a\0\x3a\x52\x66\x80\x0a\x33\x40\0\xa5\xfa\x50\x01\xdf\x9a\x3b\xd0\x01\x45\0\x2d\x1c\x52\x01\x28\xa0\x05\xa2\x80\x13\xd6\x8a\0\x28\xa0\x04\x34\x50\0\x4d\x19\xe2\x80\x0e\x82\x92\x80\x17\x34\x99\xa0\x03\x34\x9d\xe8\0\x3d\x28\xcd\x03\x0a\x28\x10\x1f\x5a\x4c\xd0\x01\x9a\x4e\xd4\0\x99\xa4\xa0\0\xd1\x9a\0\x43\x48\x69\x80\x99\xa3\xbd\0\x25\x26\x28\0\x34\x7a\x50\x03\x68\xed\x40\x07\x7a\x43\x40\x09\x45\0\x25\x21\xcd\x02\x26\xcd\x2e\x69\x80\xb4\x0a\0\x29\x47\xb5\0\x14\x7d\x68\x01\xd4\x76\xa0\x05\xed\x45\x20\x0a\x5c\xd0\0\x0d\0\xe6\x81\x8e\x14\xb4\0\x0e\xb9\xa5\xa0\x03\x34\x7d\x68\0\x3c\xd2\x50\x01\x93\xeb\x45\0\x28\xa3\xeb\x40\x07\x4e\xb4\x66\x80\x0a\x3a\x50\x02\xe6\x8c\xfa\x50\x01\x49\x40\x05\x2d\0\x20\x34\xbe\xf4\x80\x42\x68\xcd\x30\x0c\xfa\x52\x77\xa0\x03\xbd\x26\x69\0\x0a\x33\x4c\x03\x34\x94\x80\x09\xa2\x98\x06\x68\xcd\0\x1d\xe8\xcd\x20\x12\x83\x40\x05\x21\x34\0\x86\x92\x98\x05\x25\0\x25\x1f\x5a\0\x43\x45\0\x25\x26\x68\0\xcd\x25\0\x14\x94\0\x52\x67\x8a\0\x4c\xd2\x66\x81\x01\xa4\xed\x40\x13\xd1\xd6\x98\x0b\xde\x96\x80\x12\x96\x80\x0c\xe2\x97\x39\xa0\x05\xce\x28\xdd\x48\x05\xcf\x6a\x33\x40\x06\x7a\xd2\xd0\0\x3a\xd2\xd0\x03\xb3\x40\xa0\x03\x34\xe1\xef\x40\xc0\x51\x40\x84\x34\x86\x81\x8b\x49\x9a\0\x5a\x5e\x94\0\x7d\x69\x3e\x94\x80\x29\x7b\xd3\x01\x29\x71\x48\x04\xa2\x98\x0a\x0d\x14\x80\x33\x9a\x28\x01\x28\xa0\x04\xcd\x14\0\x77\xa3\xeb\x40\x06\x68\x26\x80\x13\xb5\x14\0\x52\x53\0\xa2\x90\x07\x6c\x51\x9a\0\x29\x33\x40\x05\x26\x69\x80\x80\xf3\x48\x4d\0\x14\x99\xa0\x04\xcd\x14\0\x86\x8a\0\x0d\x26\x79\xa0\x04\xa3\x3c\x50\0\x3a\xd2\x50\x20\x27\x34\xda\0\x4a\x28\x01\x01\xa4\xcd\0\x54\xff\0\x84\x83\x4c\xff\0\x9f\xaf\xfc\x86\xdf\xe1\x47\xfc\x24\x1a\x5f\xfc\xfd\x7f\xe4\x36\xff\0\x0a\xbe\x46\x47\x3a\x17\xfe\x12\x1d\x2f\xfe\x7e\xbf\xf2\x1b\x7f\x85\x1f\xf0\x90\xe9\x7f\xf3\xf5\xff\0\x90\xdb\xfc\x29\xf2\x48\x39\xd0\x7f\xc2\x41\xa5\xff\0\xcf\xd7\xfe\x43\x6f\xf0\xa3\xfe\x12\x1d\x2f\x3f\xf1\xf5\xff\0\x90\xdb\xfc\x28\xe4\x90\x73\xa1\x7f\xe1\x21\xd2\xff\0\xe7\xeb\xff\0\x21\xb7\xf8\x50\x7c\x43\xa5\xff\0\xcf\xcf\xfe\x43\x6f\xf0\xa5\xc9\x20\xe7\x41\xff\0\x09\x0e\x97\xff\0\x3f\x5f\xf9\x0d\xff\0\xc2\x81\xe2\x1d\x2f\xfe\x7e\xbf\xf2\x1b\x7f\x85\x1c\x92\x0e\x78\x8b\xff\0\x09\x16\x95\xff\0\x3f\x5f\xf9\x0d\xbf\xc2\x8f\xf8\x48\xb4\xbf\xf9\xfa\xff\0\xc8\x6d\xfe\x14\x72\x48\x39\xe2\x03\xc4\x5a\x57\xfc\xfd\x7f\xe4\x37\xff\0\x0a\x5f\xf8\x48\xf4\xaf\xf9\xfa\xff\0\xc8\x6f\xfe\x14\x72\x48\x39\xd0\x0f\x11\x69\x5f\xf3\xf5\xff\0\x90\xdf\xfc\x29\x7f\xe1\x23\xd2\xbf\xe7\xeb\xff\0\x21\xbf\xf8\x51\xc9\x20\xe7\x42\x8f\x11\xe9\x3f\xf3\xf5\xff\0\x90\xdf\xfc\x28\xff\0\x84\x93\x49\xff\0\x9f\xaf\xfc\x86\xff\0\xe1\x47\x24\x87\xcf\x10\xff\0\x84\x93\x4a\xff\0\x9f\xbf\xfc\x86\xff\0\xe1\x4b\xff\0\x09\x26\x93\xff\0\x3f\x5f\xf9\x0d\xff\0\xc2\x8e\x49\x07\x3c\x43\xfe\x12\x5d\x27\xbd\xd7\xfe\x43\x7f\xf0\xa5\xff\0\x84\x93\x48\xff\0\x9f\xbf\xfc\x86\xff\0\xe1\x47\x24\x83\x9e\x21\xff\0\x09\x26\x93\xff\0\x3f\x7f\xf9\x0d\xff\0\xc2\x93\xfe\x12\x4d\x27\xfe\x7e\xff\0\xf2\x1b\xff\0\x85\x1c\x92\x0e\x78\x87\xfc\x24\x9a\x4e\x3f\xe3\xef\xff\0\x21\xbf\xf8\x51\xff\0\x09\x26\x92\x7f\xe5\xef\xff\0\x21\xbf\xf8\x51\xc9\x20\xe7\x88\x0f\x12\x69\x3f\xf3\xf7\xff\0\x90\xdf\xfc\x29\x7f\xe1\x24\xd2\x7f\xe7\xef\xff\0\x21\xbf\xf8\x51\xc9\x20\xe7\x88\x7f\xc2\x49\xa4\xff\0\xcf\xdf\xfe\x43\x7f\xf0\xa3\xfe\x12\x5d\x27\xfe\x7e\xff\0\xf2\x1b\xff\0\x85\x1c\x92\xec\x1c\xf1\x0f\xf8\x49\x34\x9f\xf9\xfb\xff\0\xc8\x6f\xfe\x14\x7f\xc2\x49\xa4\xff\0\xcf\xdf\xfe\x43\x7f\xf0\xa3\x92\x41\xcf\x10\xff\0\x84\x93\x49\xff\0\x9f\xbf\xfc\x86\xff\0\xe1\x47\xfc\x24\x9a\x4f\xfc\xfd\xff\0\xe4\x37\xff\0\x0a\x5c\x92\x0e\x78\x89\xff\0\x09\x26\x93\xff\0\x3f\x7f\xf9\x0d\xff\0\xc2\x8f\xf8\x49\x34\x9c\xff\0\xc7\xdf\xfe\x43\x7f\xf0\xa7\xc9\x20\xe7\x88\xbf\xf0\x92\xe9\x3f\xf3\xf7\xff\0\x90\xdf\xfc\x28\xff\0\x84\x97\x49\xff\0\x9f\xaf\xfc\x86\xff\0\xe1\x4b\x92\x41\xcf\x10\xff\0\x84\x97\x49\xed\x77\xff\0\x90\xdf\xfc\x29\x3f\xe1\x24\xd2\x7f\xe7\xef\xff\0\x21\xbf\xf8\x51\xc9\x20\xe7\x88\x7f\xc2\x49\xa4\xff\0\xcf\xdf\xfe\x43\x7f\xf0\xa3\xfe\x12\x4d\x27\xfe\x7e\xff\0\xf2\x1b\xff\0\x85\x1c\x92\xec\x1c\xf1\x0f\xf8\x49\x34\x9f\xf9\xfb\xff\0\xc8\x6f\xfe\x14\x7f\xc2\x49\xa4\xff\0\xcf\xd7\xfe\x43\x7f\xf0\xa3\x92\x41\xcf\x11\x3f\xe1\x24\xd2\xb3\xff\0\x1f\x7c\x7f\xd7\x37\xff\0\x0a\x5f\xf8\x49\x34\x91\xd2\xef\xff\0\x21\xbf\xf8\x51\xc9\x20\xe7\x88\x87\xc4\x7a\x4f\xfc\xfd\xff\0\xe4\x37\xff\0\x0a\x0f\x88\xf4\xaf\xf9\xfb\xff\0\xc8\x6f\xfe\x14\x72\x48\x39\xe2\x1f\xf0\x91\xe9\x3f\xf3\xf7\xff\0\x90\xdf\xfc\x28\xff\0\x84\x8f\x49\xff\0\x9f\xbf\xfc\x86\xff\0\xe1\x4f\x92\x41\xcf\x11\x3f\xe1\x23\xd2\xbf\xe7\xef\xff\0\x21\xbf\xf8\x51\xff\0\x09\x1e\x95\xff\0\x3f\x5f\xf9\x0d\xff\0\xc2\x8e\x49\x07\x3c\x43\xfe\x12\x3d\x2b\xfe\x7e\xff\0\xf2\x1b\xff\0\x85\x07\xc4\x7a\x57\xfc\xfd\x7f\xe4\x37\xff\0\x0a\x5c\x92\xec\x1c\xf1\x0f\xf8\x48\xb4\x9f\xf9\xfb\xff\0\xc8\x6f\xfe\x14\x1f\x11\xe9\x5f\xf3\xf5\xff\0\x90\xdf\xfc\x28\xe4\x90\x73\xc4\x3f\xe1\x23\xd2\xbf\xe7\xeb\xff\0\x21\xb7\xf8\x51\xff\0\x09\x1e\x95\xff\0\x3f\x5f\xf9\x0d\xff\0\xc2\x8e\x49\x07\x3c\x44\xff\0\x84\x8b\x4a\xc7\xfc\x7d\x7f\xe4\x37\xff\0\x0a\x43\xe2\x2d\x2b\xfe\x7e\xbf\xf2\x1b\x7f\x85\x3e\x49\x07\x3c\x43\xfe\x12\x2d\x2f\xfe\x7e\xbf\xf2\x1b\xff\0\x85\x27\xfc\x24\x3a\x57\xfc\xfd\x7f\xe4\x37\xff\0\x0a\x39\x24\x1c\xf1\x0f\xf8\x48\x74\xbf\xf9\xfa\xff\0\xc8\x6d\xfe\x14\x7f\xc2\x43\xa5\xff\0\xcf\xd7\xfe\x43\x6f\xf0\xa3\x92\x41\xcf\x11\x3f\xe1\x21\xd2\xff\0\xe7\xeb\xff\0\x21\xb7\xf8\x51\xff\0\x09\x0e\x97\xff\0\x3f\x5f\xf9\x0d\xbf\xc2\x8e\x49\x07\x3c\x44\x3e\x20\xd2\xfb\x5d\x7f\xe4\x36\xff\0\x0a\x3f\xe1\x20\xd3\x3f\xe7\xeb\xff\0\x21\xb7\xf8\x51\xc9\x2e\xc1\xce\x83\xfb\x7f\x4b\xff\0\x9f\xaf\xfc\x86\xdf\xe1\x49\xff\0\x09\x06\x99\xff\0\x3f\x3f\xf9\x0d\xbf\xc2\x8e\x49\x76\x0e\x78\x87\xfc\x24\x1a\x67\xfc\xfc\xff\0\xe4\x36\xff\0\x0a\x4f\xed\xfd\x33\xfe\x7e\x7f\xf2\x1b\x7f\x85\x1c\x92\x0e\x74\x1f\xdb\xfa\x67\xfc\xfc\xff\0\xe4\x36\xff\0\x0a\x3f\xb7\xf4\xcc\x7f\xc7\xcf\xfe\x43\x6f\xf0\xa3\x92\x41\xce\x84\xfe\xde\xd3\x3f\xe7\xe7\xff\0\x21\xb7\xf8\x52\x1d\x7b\x4d\xcf\xfc\x7c\xff\0\xe4\x36\xff\0\x0a\x39\x24\x1c\xe8\x0e\xbd\xa6\xff\0\xcf\xcf\xfe\x38\xdf\xe1\x49\xfd\xbb\xa6\xff\0\xcf\xcf\xfe\x38\xdf\xe1\x47\x24\x85\xce\x83\xfb\x77\x4d\xff\0\x9f\x8f\xfc\x71\xbf\xc2\x93\xfb\x77\x4e\xff\0\x9f\x9f\xfc\x71\xbf\xc2\x8e\x49\x07\x3a\x38\xda\x2b\xa0\xc0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x02\x8a\0\x28\xa0\x0f\xff\xd9\x9f\xbf\xfc\x86\xff\0\xe1\x47\xfc\x24\x9a\x4f\xfc\xfd\xff\0\xe4\x37\xff\0\x0a\x5c\x92\x0e\x78\x89\xff\0\x09\x26\x93\xff\0\x3f\x7f\xf9\x0d\xff\0\xc2\x8f\xf8\x49\x34\x9c\xff\0\xc7\xdf\xfe\x43\x7f\xf0\xa7\xc9\x20\xe7\x88\xbf\xf0\x92\xe9\x3f\xf3\xf7\xff\0\x90\xdf\xfc\x28\xff\0\x84\x97\x49\xff\0\x9f\xaf\xfc\x86\xff\0\xe1\x4b\x92\x41\xcf\x10\xff\0\x84\x97\x49\xed\x77\xff\0\x90\xdf\xfc\x29\x3f\xe1\x24\xd2\x7f\xe7\xef\xff\0\x21\xbf\xf8\x51\xc9\x20\xe7\x88\x7f\xc2\x49\xa4\xff\0\xcf\xdf\xfe\x43\x7f\xf0\xa3\xfe\x12\x4d\x27\xfe\x7e\xff\0\xf2\x1b\xff\0\x85\x1c\x92\xec\x1c\xf1\x0f\xf8\x49\x34\x9f\xf9\xfb\xff\0\xc8\x6f\xfe\x14\x7f\xc2\x49\xa4\xff\0\xcf\xd7\xfe\x43\x7f\xf0\xa3\x92\x41\xcf\x11\x3f\xe1\x24\xd2\xb3\xff\0\x1f\x7c\x7f\xd7\x37\xff\0\x0a\x5f\xf8\x49\x34\x91\xd2\xef\xff\0\x21\xbf\xf8\x51\xc9\x20\xe7\x88\x87\xc4\x7a\x4f\xfc\xfd\xff\0\xe4\x37\xff\0\x0a\x0f\x88\xf4\xaf\xf9\xfb\xff\0\xc8\x6f\xfe\x14\x72\x48\x39\xe2\x1f\xf0\x91\xe9\x3f\xf3\xf7\xff\0\x90\xdf\xfc\x28\xff\0\x84\x8f\x49\xff\0\x9f\xbf\xfc\x86\xff\0\xe1\x4f\x92\x41\xcf\x11\x3f\xe1\x23\xd2\xbf\xe7\xef\xff\0\x21\xbf\xf8\x51\xff\0\x09\x1e\x95\xff\0\x3f\x5f\xf9\x0d\xff\0\xc2\x8e\x49\x07\x3c\x43\xfe\x12\x3d\x2b\xfe\x7e\xff\0\xf2\x1b\xff\0\x85\x07\xc4\x7a\x57\xfc\xfd\x7f\xe4\x37\xff\0\x0a\x5c\x92\xec\x1c\xf1\x0f\xf8\x48\xb4\x9f\xf9\xfb\xff\0\xc8\x6f\xfe\x14\x1f\x11\xe9\x5f\xf3\xf5\xff\0\x90\xdf\xfc\x28\xe4\x90\x73\xc4\x3f\xe1\x23\xd2\xbf\xe7\xeb\xff\0\x21\xb7\xf8\x51\xff\0\x09\x1e\x95\xff\0\x3f\x5f\xf9\x0d\xff\0\xc2\x8e\x49\x07\x3c\x44\xff\0\x84\x8b\x4a\xc7\xfc\x7d\x7f\xe4\x37\xff\0\x0a\x43\xe2\x2d\x2b\xfe\x7e\xbf\xf2\x1b\x7f\x85\x3e\x49\x07\x3c\x43\xfe\x12\x2d\x2f\xfe\x7e\xbf\xf2\x1b\xff\0\x85\x27\xfc\x24\x3a\x57\xfc\xfd\x7f\xe4\x37\xff\0\x0a\x39\x24\x1c\xf1\x0f\xf8\x48\x74\xbf\xf9\xfa\xff\0\xc8\x6d\xfe\x14\x7f\xc2\x43\xa5\xff\0\xcf\xd7\xfe\x43\x6f\xf0\xa3\x92\x41\xcf\x11\x3f\xe1\x21\xd2\xff\0\xe7\xeb\xff\0\x21\xb7\xf8\x51\xff\0\x09\x0e\x97\xff\0\x3f\x5f\xf9\x80\x0a\x28\0\xa2\x80\x0a\x28\0\xa2\x80\x0a\x28\0\xa2\x80\x0a\x28\0\xa2\x80\x0a\x28\0\xa2\x80\x0a\x28\0\xa2\x80\x0a\x28\0\xa2\x80\x0a\x28\0\xa2\x80\x0a\x28\0\xa2\x80\x0a\x28\0\xa2\x80\x0a\x28\0\xa2\x80\x0a\x28\0\xa2\x80\x0a\x28\0\xa2\x80\x0a\x28\0\xa2\x80\x0a\x28\0\xa2\x80\x0a\x28\0\xa2\x80\x0a\x28\0\xa2\x80\x3f\xff\xd9\xff\xfc\x28\xe4\x90\x73\xc4\x3f\xe1\x25\xd2\x7f\xe7\xef\xff\0\x21\xbf\xf8\x51\xff\0\x09\x2e\x93\xff\0\x3f\x7f\xf9\x0d\xff\0\xc2\x9f\x24\xbb\x07\x3c\x43\xfe\x12\x5d\x27\xfe\x7e\xff\0\xf2\x1b\xff\0\x85\x1f\xf0\x92\x69\x19\xff\0\x8f\xbf\xfc\x86\xff\0\xe1\x4b\x92\x41\xcf\x11\x7f\xe1\x25\xd2\x3f\xe7\xef\xff\0\x21\xbf\xf8\x51\xff\0\x09\x2e\x93\xff\0\x3f\x7f\xf9\x0d\xff\0\xc2\x8e\x49\x07\x3c\x43\xfe\x12\x5d\x23\xfe\x7e\xff\0\xf2\x1b\xff\0\x85\x27\xfc\x24\xba\x47\xfc\xfd\xff\0\xe4\x37\xff\0\x0a\x3d\x9c\x83\x9e\x21\xff\0\x09\x2e\x93\xff\0\x3f\x7f\xf9\x0d\xff\0\xc2\x8f\xf8\x49\x74\x9f\xf9\xfb\xff\0\xc8\x6f\xfe\x14\x72\x4b\xb0\x73\xc4\x3f\xe1\x24\xd2\x7f\xe7\xef\xff\0\x21\xbf\xf8\x52\x1f\x12\xe9\x3f\xf3\xf7\xff\0\x90\xdf\xfc\x28\xe4\x90\x73\xc4\x3f\xe1\x24\xd2\x7f\xe7\xef\xff\0\x21\xbf\xf8\x51\xff\0\x09\x26\x93\xff\0\x3f\x7f\xf9\x0d\xff\0\xc2\x8e\x49\x07\x3c\x43\xfe\x12\x4d\x27\xfe\x7e\xff\0\xf2\x1b\xff\0\x85\x27\xfc\x24\x7a\x4e\x3f\xe3\xef\xff\0\x21\xbf\xf8\x51\xc9\x20\xe7\x88\x7f\xc2\x49\xa4\xe3\xfe\x3e\xff\0\xf2\x1b\xff\0\x85\x2f\xfc\x24\x9a\x4f\xfc\xfd\x7f\xe4\x37\xff\0\x0a\x39\x24\x1c\xf1\x13\xfe\x12\x3d\x27\xfe\x7e\xff\0\xf2\x1b\xff\0\x85\x1f\xf0\x92\x69\x3f\xf3\xf7\xff\0\x90\xdf\xfc\x28\xe4\x90\x73\xc4\x3f\xe1\x23\xd2\x7f\xe7\xef\xff\0\x21\xbf\xf8\x52\x1f\x11\xe9\x27\xad\xdf\xfe\x43\x7f\xf0\xa3\x92\x41\xcf\x11\x7f\xe1\x23\xd2\x7f\xe7\xeb\xff\0\x21\xbf\xf8\x52\x7f\xc2\x47\xa5\x7f\xcf\xd7\xfe\x43\x7f\xf0\xa7\xc9\x20\xe7\x88\x7f\xc2\x47\xa5\x7f\xcf\xd7\xfe\x43\x7f\xf0\xa4\x3e\x22\xd2\xbf\xe7\xeb\xff\0\x21\xb7\xf8\x51\xc9\x20\xe7\x88\x7f\xc2\x45\xa5\x76\xba\xff\0\xc8\x6d\xfe\x14\x9f\xf0\x91\x69\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\x5a\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xa5\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff";
        $this->assertEquals($entry->getValue(), $expected);
        $this->assertEquals($entry->getText(), '55296 bytes unknown MakerNote data');

        $entry = $ifd0_0->getEntry(40960); // FlashPixVersion
        $this->assertInstanceOf('lsolesen\pel\PelEntryVersion', $entry);
        $this->assertEquals($entry->getValue(), 1);
        $this->assertEquals($entry->getText(), 'FlashPix Version 1.0');

        $entry = $ifd0_0->getEntry(40961); // ColorSpace
        $this->assertInstanceOf('lsolesen\pel\PelEntryShort', $entry);
        $this->assertEquals($entry->getValue(), 65535);
        $this->assertEquals($entry->getText(), 'Uncalibrated');

        $entry = $ifd0_0->getEntry(40962); // PixelXDimension
        $this->assertInstanceOf('lsolesen\pel\PelEntryLong', $entry);
        $this->assertEquals($entry->getValue(), 1536);
        $this->assertEquals($entry->getText(), '1536');

        $entry = $ifd0_0->getEntry(40963); // PixelYDimension
        $this->assertInstanceOf('lsolesen\pel\PelEntryLong', $entry);
        $this->assertEquals($entry->getValue(), 1024);
        $this->assertEquals($entry->getText(), '1024');

        $entry = $ifd0_0->getEntry(41495); // SensingMethod
        $this->assertInstanceOf('lsolesen\pel\PelEntryShort', $entry);
        $this->assertEquals($entry->getValue(), 2);
        $this->assertEquals($entry->getText(), 'One-chip color area sensor');

        $entry = $ifd0_0->getEntry(41728); // FileSource
        $this->assertInstanceOf('lsolesen\pel\PelEntryUndefined', $entry);
        $this->assertEquals($entry->getValue(), "\x03");
        $this->assertEquals($entry->getText(), 'DSC');

        $entry = $ifd0_0->getEntry(41729); // SceneType
        $this->assertInstanceOf('lsolesen\pel\PelEntryUndefined', $entry);
        $this->assertEquals($entry->getValue(), "\x01");
        $this->assertEquals($entry->getText(), 'Directly photographed');

        $entry = $ifd0_0->getEntry(41985); // CustomRendered
        $this->assertInstanceOf('lsolesen\pel\PelEntryShort', $entry);
        $this->assertEquals($entry->getValue(), 0);
        $this->assertEquals($entry->getText(), 'Normal process');

        $entry = $ifd0_0->getEntry(41986); // ExposureMode
        $this->assertInstanceOf('lsolesen\pel\PelEntryShort', $entry);
        $this->assertEquals($entry->getValue(), 0);
        $this->assertEquals($entry->getText(), 'Auto exposure');

        $entry = $ifd0_0->getEntry(41987); // WhiteBalance
        $this->assertInstanceOf('lsolesen\pel\PelEntryShort', $entry);
        $this->assertEquals($entry->getValue(), 1);
        $this->assertEquals($entry->getText(), 'Manual white balance');

        $entry = $ifd0_0->getEntry(41989); // FocalLengthIn35mmFilm
        $this->assertInstanceOf('lsolesen\pel\PelEntryShort', $entry);
        $this->assertEquals($entry->getValue(), 67);
        $this->assertEquals($entry->getText(), '67');

        $entry = $ifd0_0->getEntry(41990); // SceneCaptureType
        $this->assertInstanceOf('lsolesen\pel\PelEntryShort', $entry);
        $this->assertEquals($entry->getValue(), 0);
        $this->assertEquals($entry->getText(), 'Standard');

        $entry = $ifd0_0->getEntry(41992); // Contrast
        $this->assertInstanceOf('lsolesen\pel\PelEntryShort', $entry);
        $this->assertEquals($entry->getValue(), 0);
        $this->assertEquals($entry->getText(), 'Normal');

        $entry = $ifd0_0->getEntry(41993); // Saturation
        $this->assertInstanceOf('lsolesen\pel\PelEntryShort', $entry);
        $this->assertEquals($entry->getValue(), 0);
        $this->assertEquals($entry->getText(), 'Normal');

        $entry = $ifd0_0->getEntry(41994); // Sharpness
        $this->assertInstanceOf('lsolesen\pel\PelEntryShort', $entry);
        $this->assertEquals($entry->getValue(), 0);
        $this->assertEquals($entry->getText(), 'Normal');

        $entry = $ifd0_0->getEntry(41996); // SubjectDistanceRange
        $this->assertInstanceOf('lsolesen\pel\PelEntryShort', $entry);
        $this->assertEquals($entry->getValue(), 1);
        $this->assertEquals($entry->getText(), 'Macro');

        $entry = $ifd0_0->getEntry(42240); // Gamma
        $this->assertInstanceOf('lsolesen\pel\PelEntryRational', $entry);
        $this->assertEquals($entry->getValue(), array(
            0 => 22,
            1 => 10
        ));
        $this->assertEquals($entry->getText(), '22/10');

        /* Sub IFDs of $ifd0_0. */
        $this->assertEquals(count($ifd0_0->getSubIfds()), 1);
        $ifd0_0_0 = $ifd0_0->getSubIfd(4); // IFD Interoperability
        $this->assertInstanceOf('lsolesen\pel\PelIfd', $ifd0_0_0);

        /* Start of IDF $ifd0_0_0. */
        $this->assertEquals(count($ifd0_0_0->getEntries()), 2);

        $entry = $ifd0_0_0->getEntry(1); // InteroperabilityIndex
        $this->assertInstanceOf('lsolesen\pel\PelEntryAscii', $entry);
        $this->assertEquals($entry->getValue(), 'R03');
        $this->assertEquals($entry->getText(), 'R03');

        $entry = $ifd0_0_0->getEntry(2); // InteroperabilityVersion
        $this->assertInstanceOf('lsolesen\pel\PelEntryVersion', $entry);
        $this->assertEquals($entry->getValue(), 1);
        $this->assertEquals($entry->getText(), 'Interoperability Version 1.0');

        /* Sub IFDs of $ifd0_0_0. */
        $this->assertEquals(count($ifd0_0_0->getSubIfds()), 0);

        $this->assertEquals($ifd0_0_0->getThumbnailData(), '');

        /* Next IFD. */
        $ifd0_0_1 = $ifd0_0_0->getNextIfd();
        $this->assertNull($ifd0_0_1);
        /* End of IFD $ifd0_0_0. */

        $this->assertEquals($ifd0_0->getThumbnailData(), '');

        /* Next IFD. */
        $ifd0_1 = $ifd0_0->getNextIfd();
        $this->assertNull($ifd0_1);
        /* End of IFD $ifd0_0. */

        $this->assertEquals($ifd0->getThumbnailData(), '');

        /* Next IFD. */
        $ifd1 = $ifd0->getNextIfd();
        $this->assertInstanceOf('lsolesen\pel\PelIfd', $ifd1);
        /* End of IFD $ifd0. */

        /* Start of IDF $ifd1. */
        $this->assertEquals(count($ifd1->getEntries()), 4);

        $entry = $ifd1->getEntry(259); // Compression
        $this->assertInstanceOf('lsolesen\pel\PelEntryShort', $entry);
        $this->assertEquals($entry->getValue(), 6);
        $this->assertEquals($entry->getText(), 'JPEG compression');

        $entry = $ifd1->getEntry(282); // XResolution
        $this->assertInstanceOf('lsolesen\pel\PelEntryRational', $entry);
        $this->assertEquals($entry->getValue(), array(
            0 => 72,
            1 => 1
        ));
        $this->assertEquals($entry->getText(), '72/1');

        $entry = $ifd1->getEntry(283); // YResolution
        $this->assertInstanceOf('lsolesen\pel\PelEntryRational', $entry);
        $this->assertEquals($entry->getValue(), array(
            0 => 72,
            1 => 1
        ));
        $this->assertEquals($entry->getText(), '72/1');

        $entry = $ifd1->getEntry(296); // ResolutionUnit
        $this->assertInstanceOf('lsolesen\pel\PelEntryShort', $entry);
        $this->assertEquals($entry->getValue(), 2);
        $this->assertEquals($entry->getText(), 'Inch');

        /* Sub IFDs of $ifd1. */
        $this->assertEquals(count($ifd1->getSubIfds()), 0);

        $thumb_data = file_get_contents(dirname(__FILE__) . '/pentax-istDS-thumb.jpg');
        $this->assertEquals($ifd1->getThumbnailData(), $thumb_data);

        /* Next IFD. */
        $ifd2 = $ifd1->getNextIfd();
        $this->assertNull($ifd2);
        /* End of IFD $ifd1. */

        $this->assertTrue(count(Pel::getExceptions()) == 0);
    }
}
