<?php
// vim: set ts=4 sw=4 sts=4 et:

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace Includes\Utils;

/**
 * Database
 *
 */
abstract class Database extends \Includes\Utils\AUtils
{
    /**
     * SQL files must contain this table prefix placeholder
     */
    const TABLE_PREFIX_PLACEHOLDER = '%%XC%%_';

    /**
     * DB handler
     *
     * @var \PDO
     */
    protected static $handler;

    /**
     * Database connection options
     *
     * @var array
     */
    protected static $dbOptions;

    /**
     * Setter method for $dbOptions. Once tries to connect and return connection object
     *
     * @return \PDO
     */
    public static function setDbOptions($options)
    {
        static::$dbOptions = $options;

        return static::getHandler();
    }

    /**
     * Reset method for $this->dbOptions
     *
     * @return void
     */
    public static function resetDbOptions()
    {
        static::$dbOptions = null;
    }

    /**
     * Return array of credentials to connect to DB
     *
     * @param bool $fullList add or not the additional fields
     *
     * @return array
     */
    public static function getConnectionParams($fullList = false)
    {
        $options = static::getDbOptions();

        $dsnFields = array(
            'host'        => 'hostspec',
            'port'        => 'port',
            'unix_socket' => 'socket',
            'dbname'      => 'database',
        );

        foreach ($dsnFields as $pdoOption => $lcOption) {
            if (!empty($options[$lcOption])) {
                $dsnFields[$pdoOption] = $options[$lcOption];
            } else {
                unset($dsnFields[$pdoOption]);
            }
        }

        if ($fullList) {
            $dsnFields['username'] = static::getUsername();
            $dsnFields['user'] = $dsnFields['username'];
            $dsnFields['password'] = static::getPassword();
        }

        return $dsnFields;
    }

    /**
     * Prepare MySQL connection string
     *
     * @return string
     */
    public static function getConnectionString()
    {
        return 'mysql:' . Converter::buildQuery(static::getConnectionParams(), '=', ';');
    }

    /**
     * Getter method for $this->dbOptions
     *
     * @return array
     */
    protected static function getDbOptions($name = null)
    {
        return ArrayManager::getIndex(
            static::$dbOptions ?: ConfigParser::getOptions(array('database_details')),
            $name
        );
    }

    /**
     * Return name of database user
     *
     * @return string
     */
    protected static function getUsername()
    {
        return static::getDbOptions('username');
    }

    /**
     * Return password of database user
     *
     * @return string
     */
    protected static function getPassword()
    {
        return static::getDbOptions('password');
    }

    /**
     * Return list of the \PDO connection options
     *
     * @return array
     */
    protected static function getConnectionFlags()
    {
        return array(
            \PDO::ATTR_AUTOCOMMIT => true,
            \PDO::ATTR_ERRMODE    => \PDO::ERRMODE_EXCEPTION,
            \PDO::ATTR_PERSISTENT => false,
        );
    }

    /**
     * Connect to database
     *
     * @return \PDO
     */
    protected static function connectToDb()
    {
        return new \PDO(
            static::getConnectionString(),
            static::getUsername(),
            static::getPassword(),
            static::getConnectionFlags()
        );
    }

    /**
     * Return \PDO database handler
     *
     * @return \PDO
     */
    protected static function getHandler()
    {
        if (!isset(static::$handler)) {
            static::$handler = static::connectToDb();
        }

        return static::$handler;
    }

    /**
     * Execute SQL query and return the PDO statement object
     *
     * @param string $sql    SQL query to execute
     * @param array  $params Query params
     *
     * @return \PDOStatement
     */
    public static function executeStatement($sql, array $params = array())
    {
        $statement = static::getHandler()->prepare($sql);
        $statement->execute($params);

        return $statement;
    }

    /**
     * Perform SQL query (return araay of records)
     *
     * @param string  $sql    SQL query to execute
     * @param array   $params Query params
     * @param integer $flags  \PDO fetch option
     *
     * @return array
     */
    public static function fetchAll($sql, array $params = array(), $flags = \PDO::FETCH_ASSOC)
    {
        return static::executeStatement($sql, $params)->fetchAll($flags);
    }

    /**
     * Perform SQL query (single value)
     *
     * @param string $sql    SQL query to execute
     * @param array  $params Query params
     *
     * @return string
     */
    public static function fetchColumn($sql, array $params = array())
    {
        return static::executeStatement($sql, $params)->fetchColumn();
    }

    /**
     * Perform parameterized SQL query and return the flag (success or not)
     *
     * @param string $sql    SQL query to execute
     * @param array  $params Query params
     *
     * @return void
     */
    public static function execute($sql, array $params = array())
    {
        static::executeStatement($sql, $params);
    }

    /**
     * Perform SQL query
     *
     * @param string $sql    SQL query to execute
     * @param array  $params query params
     *
     * @return bool
     */
    public static function exec($sql)
    {
        return static::getHandler()->exec($sql);
    }

    /**
     * Get the database version
     *
     * @return string
     */
    public static function getDbVersion()
    {
        return static::getHandler()->getAttribute(\PDO::ATTR_SERVER_VERSION);
    }

    /**
     * Checks InnoDB support
     *
     * @return boolean
     */
    public static function isInnoDBSupported()
    {
        $innodbFound = false;

        $result = static::getHandler()->query('SHOW ENGINES');

        if ($result) {

            foreach ($result as $row) {

                if (0 === strcasecmp('InnoDB', $row['Engine'])) {
                    $innodbFound = true;
                    break;
                }
            }
        }

        return $innodbFound;
    }

    /**
     * Execute a set of SQL queries from file
     *
     * :FIXME: must be completely revised
     *
     * @param string $fileName Name of SQL-file
     * @param boolean $verbose Display uploading progress flag OPTIONAL
     *
     * @return boolean
     * @throws \InvalidArgumentException
     */
    public static function uploadSQLFromFile($fileName, $verbose = false)
    {
        $result = false;

        if (false === FileManager::isFileReadable($fileName)) {

            throw new \InvalidArgumentException(
                sprintf('SQL file \'%s\' not found or is not readable', $fileName)
            );

        } else {

            $fp = fopen($fileName, 'rb');

            $sql = '';

            $result = true;

            static::exec('SET NAMES utf8');

            while ($result && !feof($fp)) {

                $c = '';

                // Read SQL statement from file
                do {
                    $c .= fgets($fp, 1024);
                    $endPos = strlen($c) - 1;

                } while (substr($c, $endPos) != PHP_EOL && !feof($fp));

                $c = rtrim($c);

                // Skip comments
                if (substr($c, 0, 1) == '#' || substr($c, 0, 2) == '--') {
                    continue;
                }

                // Parse SQL statement

                $sql .= $c;

                if (substr($sql, -1) == ';') {

                    $sql = substr($sql, 0, strlen($sql) - 1);
                    $sql = str_replace(static::TABLE_PREFIX_PLACEHOLDER, static::getTablesPrefix(), $sql);

                    // Execute SQL query
                    try {

                        static::getHandler()->beginTransaction();
                        $result = (false !== static::exec($sql));

                        if ($result) {
                            static::getHandler()->commit();

                        } else {
                            static::getHandler()->rollBack();
                        }

                        if ($verbose) {
                            Operator::flush('. ');
                        }

                    } catch (\PDOException $e) {

                        static::getHandler()->rollBack();

                        $result = false;

                        Operator::flush(LC_EOL . $e->getMessage());
                    }
                    $sql = '';
                }
            }

            fclose($fp);
        }

        return $result;
    }

    /**
     * Return tables prefix in the DB
     *
     * @return string
     */
    public static function getTablesPrefix()
    {
        return ConfigParser::getOptions(array('database_details', 'table_prefix'));
    }
}
